// app/_layout.tsx
// Add navigator polyfill at the very top
if (typeof navigator === "undefined") {
  // (global as any) is used to bypass TypeScript strictness
  (global as any).navigator = { userAgent: "ReactNative" };
}

import * as Notifications from "expo-notifications";
import FlashMessage, { showMessage } from "react-native-flash-message";
import { AppState } from "react-native";

// 안드로이드용 알림 채널 설정 함수
const setupNotificationChannel = async () => {
  console.log("setupNotificationChannel 호출됨");
  if (Platform.OS === "android") {
    await Notifications.setNotificationChannelAsync("default", {
      name: "default",
      importance: Notifications.AndroidImportance.MAX,
      vibrationPattern: [0, 250, 250, 250],
      lightColor: "#FF231F7C",
      enableVibrate: true,
      showBadge: true,
    });
    console.log("안드로이드 알림 채널 설정 완료");
  }
};

// 알림 핸들러 설정 (포그라운드에서도 알림 표시)
Notifications.setNotificationHandler({
  handleNotification: async () => {
    console.log("알림 핸들러 호출됨");
    return {
      shouldShowAlert: true, // 포그라운드에서도 알림 표시
      shouldPlaySound: true,
      shouldSetBadge: true,
      priority: Notifications.AndroidNotificationPriority.MAX,
    };
  },
});

import { useFonts } from "expo-font";
import { Stack } from "expo-router";
import * as SplashScreen from "expo-splash-screen";
import { useEffect, useRef } from "react";
import "react-native-reanimated";
import {
  StatusBar,
  setStatusBarBackgroundColor,
  setStatusBarStyle,
  setStatusBarTranslucent,
} from "expo-status-bar";
import * as Device from "expo-device";
import Constants from "expo-constants";
import { Platform } from "react-native";

import {
  SafeAreaProvider,
  useSafeAreaInsets,
} from "react-native-safe-area-context";
import { Colors } from "@/assets/styles/App.styles";
import { MD3LightTheme, PaperProvider } from "react-native-paper";
import { NativeBaseProvider } from "native-base";
import { GestureHandlerRootView } from "react-native-gesture-handler";
import { usePathname } from "expo-router";
import { QueryClient, QueryClientProvider } from "@tanstack/react-query";
import useAuthStore from "@/app/store/authStore";
import api from "@/app/lib/axios"; // Adjust this import if your API client is located elsewhere

// Prevent the splash screen from auto-hiding before asset loading is complete.
SplashScreen.preventAutoHideAsync();

// React Query 클라이언트 설정 수정 - 불필요한 API 호출 감소
const queryClient = new QueryClient({
  defaultOptions: {
    queries: {
      staleTime: 30000, // 기본 30초 캐싱
      refetchOnWindowFocus: false, // 윈도우 포커스 시 자동 리패치 비활성화
      retry: 1, // 실패 시 1번만 재시도
      refetchOnMount: true, // 마운트 시 리패치 (always가 아닌 true로 설정)
    },
  },
});

export default function RootLayout() {
  console.log("RootLayout 렌더됨");
  const { top, bottom, left, right } = useSafeAreaInsets();
  const [loaded] = useFonts({
    SpaceMono: require("../assets/fonts/SpaceMono-Regular.ttf"),
    popregular: require("../assets/fonts/popregular.ttf"),
    popsemibold: require("../assets/fonts/popsemibold.ttf"),
    popbold: require("../assets/fonts/popbold.ttf"),
    workregular: require("../assets/fonts/workregular.ttf"),
    workmed: require("../assets/fonts/workmed.ttf"),
    worksemibold: require("../assets/fonts/worksemibold.ttf"),
    workbold: require("../assets/fonts/workbold.ttf"),
    montregular: require("../assets/fonts/Montserrat-Regular.ttf"),
    montmedium: require("../assets/fonts/Montserrat-Medium.ttf"),
    montsemibold: require("../assets/fonts/Montserrat-SemiBold.ttf"),
    montbold: require("../assets/fonts/Montserrat-Bold.ttf"),
    montitalic: require("../assets/fonts/Montserrat-Italic.ttf"),
    montmediumitalic: require("../assets/fonts/Montserrat-MediumItalic.ttf"),
    montsemibolditalic: require("../assets/fonts/Montserrat-SemiBoldItalic.ttf"),
    montbolditalic: require("../assets/fonts/Montserrat-BoldItalic.ttf"),
    TwemojiMozilla: require("../assets/fonts/TwemojiMozilla.ttf"),
  });

  // 알림 응답을 처리하기 위한 ref
  const notificationListener = useRef<Notifications.Subscription | null>(null);
  const responseListener = useRef<Notifications.Subscription | null>(null);

  // 푸시 알림 토큰 등록 함수
  const registerForPushNotificationsAsync = async () => {
    let token;

    // 실제 기기인지 확인 (시뮬레이터에서는 푸시 알림이 작동하지 않음)
    if (Device.isDevice) {
      // 권한 요청
      const { status: existingStatus } =
        await Notifications.getPermissionsAsync();
      let finalStatus = existingStatus;

      if (existingStatus !== "granted") {
        const { status } = await Notifications.requestPermissionsAsync();
        finalStatus = status;
      }

      if (finalStatus !== "granted") {
        console.log("알림 권한이 거부되었습니다!");
        return;
      }

      // Expo 푸시 토큰 가져오기
      try {
        token = (
          await Notifications.getExpoPushTokenAsync({
            projectId: "28162de2-ef64-46cd-bb61-d3fde2b5f69e",
          })
        ).data;

        console.log("Push 토큰:", token);

        // 백엔드에 토큰 전송
        try {
          const authState = useAuthStore.getState();
          // 사용자가 로그인한 경우에만 토큰 전송
          if (authState.isAuthenticated) {
            await api.post("/device-token/register", { token });
            console.log("디바이스 토큰이 서버에 등록되었습니다.");
          } else {
            console.log(
              "사용자가 인증되지 않았습니다. 토큰 등록을 건너뜁니다."
            );
          }
        } catch (error) {
          console.error("디바이스 토큰 등록 실패:", error);
        }
      } catch (error) {
        console.error("푸시 토큰 가져오기 실패:", error);
      }
    } else {
      console.log("실제 기기에서만 푸시 알림을 사용할 수 있습니다!");
    }

    // Android에서 추가 설정
    if (Platform.OS === "android") {
      Notifications.setNotificationChannelAsync("default", {
        name: "default",
        importance: Notifications.AndroidImportance.MAX,
        vibrationPattern: [0, 250, 250, 250],
        lightColor: "#FF231F7C",
      });
    }

    return token;
  };

  useEffect(() => {
    const subscription = AppState.addEventListener("change", (nextAppState) => {
      if (nextAppState === "active") {
        console.log("앱이 foreground로 전환됨 - notifications 쿼리 무효화");
        queryClient.invalidateQueries({ queryKey: ["notifications"] });
      }
    });
    return () => {
      subscription.remove();
    };
  }, []);

  // 새로운 useEffect 추가: 푸시 토큰 및 채널 설정, 리스너 등록
  useEffect(() => {
    const initPushNotifications = async () => {
      // Check if running on a physical device
      if (!Device.isDevice) {
        console.log("Push notifications require a physical device.");
        return;
      }

      // Request permissions
      const { status: existingStatus } =
        await Notifications.getPermissionsAsync();
      let finalStatus = existingStatus;
      if (existingStatus !== "granted") {
        const { status } = await Notifications.requestPermissionsAsync();
        finalStatus = status;
      }

      if (finalStatus !== "granted") {
        console.log("Permission for push notifications not granted");
        return;
      }

      try {
        // Retrieve the Expo push token
        const tokenData = await Notifications.getExpoPushTokenAsync({
          projectId: "28162de2-ef64-46cd-bb61-d3fde2b5f69e",
        });
        console.log("Push token:", tokenData.data);
      } catch (error) {
        console.error("Error getting push token:", error);
      }

      // For Android, set the notification channel
      if (Platform.OS === "android") {
        await Notifications.setNotificationChannelAsync("default", {
          name: "default",
          importance: Notifications.AndroidImportance.MAX,
          vibrationPattern: [0, 250, 250, 250],
          lightColor: "#FF231F7C",
        });
      }

      // Set up additional listeners for testing purposes
      const testNotificationListener =
        Notifications.addNotificationReceivedListener((notification) => {
          console.log("Notification received:", notification);
        });

      const testResponseListener =
        Notifications.addNotificationResponseReceivedListener((response) => {
          console.log("Notification response:", response);
        });

      // Cleanup listeners on unmount
      return () => {
        Notifications.removeNotificationSubscription(testNotificationListener);
        Notifications.removeNotificationSubscription(testResponseListener);
      };
    };

    initPushNotifications();
  }, []);

  // Now, existing hook for authentication and splash screen
  useEffect(() => {
    useAuthStore.getState().initAuth();
  }, []);

  useEffect(() => {
    if (loaded) {
      SplashScreen.hideAsync();
    }
  }, [loaded]);

  useEffect(() => {
    setStatusBarStyle("light", true);
    setStatusBarBackgroundColor(Colors.primary, true);
    setStatusBarTranslucent(true);
  }, []);

  // 새로운 useEffect: 사용자가 로그인한 후에 푸시 토큰 등록
  const isAuthenticated = useAuthStore((state) => state.isAuthenticated);
  useEffect(() => {
    if (isAuthenticated) {
      registerForPushNotificationsAsync();
    }
  }, [isAuthenticated]);

  const pathname = usePathname();
  const shouldApplyPadding = !pathname.includes("pages/home/chatPage");

  // 포그라운드 알림 수신 리스너 추가
  useEffect(() => {
    console.log("포그라운드 알림 리스너 useEffect 호출됨");
    // 알림 채널 설정 전에 로그
    setupNotificationChannel();

    const foregroundSubscription =
      Notifications.addNotificationReceivedListener((notification) => {
        console.log("포그라운드 알림 수신 (_layout):", notification);
        try {
          const { title, body } = notification.request.content;
          showMessage({
            message: title || "새 알림",
            description: body || "",
            type: "info",
            duration: 5000,
            floating: true,
            icon: "info",
            backgroundColor: "#4D8FAC",
            color: "#FFFFFF",
          });
        } catch (error) {
          console.error("포그라운드 알림 표시 오류:", error);
        }
      });

    return () => {
      if (foregroundSubscription) {
        Notifications.removeNotificationSubscription(foregroundSubscription);
      }
    };
  }, []);

  if (!loaded) {
    return null;
  }

  const paperTheme = {
    ...MD3LightTheme,
    colors: {
      ...MD3LightTheme.colors,
      primary: "#7AB52E",
      secondary: "#2E4D00",
      error: "#E63946",
      background: "#FFFFFF",
      surface: "#FFFFFF",
    },
  };

  // FCM 토큰 서버 등록 함수
  const registerFcmTokenWithServer = async (fcmToken: string) => {
    try {
      const response = await api.post("/device-tokens/register", {
        device_token: fcmToken,
        device_type: Platform.OS, // 'ios' 또는 'android'
      });

      console.log("FCM 토큰 등록 성공:", response.data);
    } catch (error) {
      console.error("FCM 토큰 등록 실패:", error);
    }
  };

  return (
    <QueryClientProvider client={queryClient}>
      <SafeAreaProvider
        style={{
          backgroundColor: Colors.primary,
          ...(shouldApplyPadding && {
            paddingTop: top,
            paddingBottom: bottom,
            paddingLeft: left,
            paddingRight: right,
          }),
        }}
      >
        <PaperProvider theme={paperTheme}>
          <GestureHandlerRootView>
            <NativeBaseProvider>
              <StatusBar
                style="light"
                backgroundColor={Colors.primary}
                animated={true}
                translucent={true}
              />
              <Stack
                screenOptions={{
                  headerShown: false,
                  contentStyle: { backgroundColor: "white" },
                }}
              ></Stack>
              <FlashMessage position="top" />
            </NativeBaseProvider>
          </GestureHandlerRootView>
        </PaperProvider>
      </SafeAreaProvider>
    </QueryClientProvider>
  );
}
