// app/api/endpoints/friends.js
import api from "@/app/lib/axios";

// 최대 재시도 횟수
const MAX_RETRIES = 2;

// API 요청 재시도 함수
const fetchWithRetry = async (apiCall, retries = 0) => {
  try {
    return await apiCall();
  } catch (error) {
    if (retries < MAX_RETRIES) {
      // 500ms 후 재시도 (지수 백오프: 500ms, 1000ms, 2000ms...)
      const delay = 500 * Math.pow(2, retries);
      console.log(
        `Retrying API call after ${delay}ms (retry ${
          retries + 1
        }/${MAX_RETRIES})`
      );
      await new Promise((resolve) => setTimeout(resolve, delay));
      return fetchWithRetry(apiCall, retries + 1);
    }
    throw error;
  }
};

export const getFriendships = async (token) => {
  try {
    const apiCall = () =>
      api.get("/friendships/", {
        headers: { Authorization: `Bearer ${token}` },
      });

    const response = await fetchWithRetry(apiCall);
    return response.data || [];
  } catch (error) {
    console.error("Error fetching friendships:", error.message);
    if (error.response && error.response.status === 404) {
      return [];
    }
    throw error;
  }
};

export const getMessageThreads = async (token) => {
  try {
    const apiCall = () =>
      api.get("/messages/threads", {
        headers: { Authorization: `Bearer ${token}` },
      });

    const response = await fetchWithRetry(apiCall);
    return response.data || [];
  } catch (error) {
    // 로그에 오류를 기록하지만 애플리케이션 흐름은 중단하지 않음
    console.warn("Error fetching message threads:", error.message);
    // 404 또는 다른 오류가 발생해도 빈 배열을 반환하여 애플리케이션이 계속 작동하도록 함
    return [];
  }
};

// 추가: 친구 요청 수락
export const acceptFriendship = async (friend_id, token) => {
  try {
    const apiCall = () =>
      api.put(`/friendships/accept/${friend_id}`, null, {
        headers: { Authorization: `Bearer ${token}` },
      });

    const response = await fetchWithRetry(apiCall);
    return response.data;
  } catch (error) {
    console.error("Error accepting friendship:", error.message);
    throw error;
  }
};

// 추가: 친구 요청 거절
export const rejectFriendship = async (friend_id, token) => {
  try {
    const apiCall = () =>
      api.put(`/friendships/reject/${friend_id}`, null, {
        headers: { Authorization: `Bearer ${token}` },
      });

    const response = await fetchWithRetry(apiCall);
    return response.data;
  } catch (error) {
    console.error("Error rejecting friendship:", error.message);
    throw error;
  }
};

// 추가: 친구 요청 보내기
export const requestFriendship = async (id, token) => {
  try {
    const apiCall = () =>
      api.post(
        `/friendships/request/${id}`,
        {},
        {
          headers: { Authorization: `Bearer ${token}` },
        }
      );

    const response = await fetchWithRetry(apiCall);
    return response.data;
  } catch (error) {
    console.error("Error requesting friendship:", error.message);
    throw error;
  }
};
