// app/api/endpoints/messages.js
import api from "@/app/lib/axios";

// 최대 재시도 횟수
const MAX_RETRIES = 2;

// API 요청 재시도 함수
const fetchWithRetry = async (apiCall, retries = 0) => {
  try {
    return await apiCall();
  } catch (error) {
    if (retries < MAX_RETRIES) {
      // 500ms 후 재시도 (지수 백오프: 500ms, 1000ms, 2000ms...)
      const delay = 500 * Math.pow(2, retries);
      console.log(
        `Retrying API call after ${delay}ms (retry ${
          retries + 1
        }/${MAX_RETRIES})`
      );
      await new Promise((resolve) => setTimeout(resolve, delay));
      return fetchWithRetry(apiCall, retries + 1);
    }
    throw error;
  }
};

export const getConversation = async (friendId, token) => {
  try {
    const apiCall = () =>
      api.get(`/messages/conversation/${friendId}`, {
        headers: { Authorization: `Bearer ${token}` },
      });

    const response = await fetchWithRetry(apiCall);
    return response.data || [];
  } catch (error) {
    // 오류 로깅 및 빈 배열 반환
    console.warn(
      `Error fetching conversation with friendId ${friendId}:`,
      error.message
    );
    return [];
  }
};

export const markMessagesAsRead = async (senderId, token) => {
  try {
    const apiCall = () =>
      api.put(
        `/messages/mark-read/${senderId}`,
        {},
        {
          headers: { Authorization: `Bearer ${token}` },
        }
      );

    const response = await fetchWithRetry(apiCall);
    return response.data;
  } catch (error) {
    console.warn(
      `Error marking messages as read from ${senderId}:`,
      error.message
    );
    // Even if it fails, return success to prevent UI disruption
    return { success: true };
  }
};

export const getUnreadMessages = async (token) => {
  try {
    const apiCall = () =>
      api.get(`/messages/unread`, {
        headers: { Authorization: `Bearer ${token}` },
      });

    const response = await fetchWithRetry(apiCall);
    return response.data || [];
  } catch (error) {
    console.warn("Error fetching unread messages:", error.message);
    return [];
  }
};
