import api from "@/app/lib/axios";

// 장바구니 조회 함수 - 타임아웃 지원 추가
export const fetchShoppingCart = async (options = {}) => {
  try {
    const { signal, timeout = 5000 } = options;

    // 시간 제한이 있는 요청 생성
    let timeoutId;
    const timeoutPromise = new Promise((_, reject) => {
      timeoutId = setTimeout(() => {
        reject(new Error("Shopping cart fetch timeout"));
      }, timeout);
    });

    // 실제 API 요청
    const fetchPromise = api.get("/shopping-cart/", { signal });

    // Promise.race로 둘 중 먼저 완료되는 것 반환
    const response = await Promise.race([fetchPromise, timeoutPromise]);

    // 타임아웃 취소
    clearTimeout(timeoutId);

    return response.data;
  } catch (error) {
    // AbortError 또는 타임아웃은 특별 처리
    if (error.name === "AbortError" || error.message.includes("timeout")) {
      console.warn("Shopping cart fetch aborted:", error.message);
      return { items: [], aborted: true };
    }

    // datetime_type 오류 처리
    if (
      error?.response?.data?.detail &&
      typeof error.response.data.detail === "string" &&
      error.response.data.detail.includes("datetime_type")
    ) {
      console.warn(
        "Shopping cart fetch encountered datetime_type error:",
        error.message
      );
      return { items: [], datetime_error: true };
    }

    // 그 외 오류 - 빈 장바구니 반환하고 로그 기록
    console.error("Shopping cart fetch failed:", error.message);
    return { items: [], error: error.message };
  }
};

// 장바구니 아이템 추가 함수 - 타임아웃 지원 추가
export const addCartItem = async (productId, quantity, options = {}) => {
  try {
    const { signal, timeout = 5000, size, variant_id } = options;

    // 수량이 0인 경우 자동으로 제거 처리
    if (quantity === 0) {
      return { success: true, local_only: true };
    }

    // 시간 제한이 있는 요청 생성
    let timeoutId;
    const timeoutPromise = new Promise((_, reject) => {
      timeoutId = setTimeout(() => {
        reject(new Error("Shopping cart add timeout"));
      }, timeout);
    });

    // 요청 페이로드 구성
    const payload = {
      product_id: productId,
      quantity: quantity,
    };

    // 사이즈 정보가 있으면 추가
    if (size) {
      payload.size = size;
      console.log(`Adding size (${size}) to cart request payload`);
    }

    // variant_id가 있으면 추가
    if (variant_id) {
      payload.variant_id = variant_id;
    }

    console.log(
      `[Shopping Cart Add] Request payload:`,
      JSON.stringify(payload, null, 2)
    );

    // 실제 API 요청
    const fetchPromise = api.post("/shopping-cart/add", payload, { signal });

    // Promise.race로 둘 중 먼저 완료되는 것 반환
    const response = await Promise.race([fetchPromise, timeoutPromise]);

    // 타임아웃 취소
    clearTimeout(timeoutId);

    return response.data;
  } catch (error) {
    // AbortError 또는 타임아웃은 특별 처리
    if (error.name === "AbortError" || error.message.includes("timeout")) {
      console.warn("Shopping cart add aborted:", error.message);
      return { success: true, local_only: true, aborted: true };
    }

    // 그 외 오류 - 로컬 처리만 하고 계속 진행
    console.error("Shopping cart add failed:", error.message);
    if (error.response) {
      console.error("Response status:", error.response.status);
      console.error(
        "Response data:",
        JSON.stringify(error.response.data, null, 2)
      );
    }
    return { success: true, local_only: true, error: error.message };
  }
};

// 장바구니 아이템 제거 함수 - 타임아웃 지원 추가
export const removeCartItem = async (productId, quantity, options = {}) => {
  try {
    const { signal, timeout = 5000, size } = options;

    // 시간 제한이 있는 요청 생성
    let timeoutId;
    const timeoutPromise = new Promise((_, reject) => {
      timeoutId = setTimeout(() => {
        reject(new Error("Shopping cart remove timeout"));
      }, timeout);
    });

    console.log(`Attempting to remove item ${productId} from cart`);

    // quantity를 0으로 설정하여 아이템 제거 요청
    const payload = {
      product_id: productId,
      quantity: 0, // 수량을 0으로 설정하여 제거 요청
    };

    // 사이즈 정보가 있으면 추가
    if (size) {
      payload.size = size;
      console.log(`Including size ${size} for product ${productId} removal`);
    }

    console.log(
      `[Shopping Cart Remove] Request payload:`,
      JSON.stringify(payload, null, 2)
    );

    // 실제 API 요청 - quantity=0으로 add 엔드포인트 호출
    const fetchPromise = api.post("/shopping-cart/add", payload, { signal });

    // Promise.race로 둘 중 먼저 완료되는 것 반환
    const response = await Promise.race([fetchPromise, timeoutPromise]);

    // 타임아웃 취소
    clearTimeout(timeoutId);

    console.log(`Successfully removed item from cart:`, response.data);
    return response.data;
  } catch (error) {
    // AbortError 또는 타임아웃은 특별 처리
    if (error.name === "AbortError" || error.message.includes("timeout")) {
      console.warn("Shopping cart remove aborted:", error.message);
      return { success: true, local_only: true, aborted: true };
    }

    // 그 외 오류 - 로컬 처리만 하고 계속 진행
    console.error("Shopping cart remove failed:", error.message);
    if (error.response) {
      console.error("Response status:", error.response.status);
      console.error(
        "Response data:",
        JSON.stringify(error.response.data, null, 2)
      );
    }
    return { success: true, local_only: true, error: error.message };
  }
};

// 장바구니 아이템 업데이트 함수 - 타임아웃 지원 추가
export const updateCartItem = async (productId, quantity, options = {}) => {
  try {
    const { signal, timeout = 5000, size, variant_id } = options;

    // 시간 제한이 있는 요청 생성
    let timeoutId;
    const timeoutPromise = new Promise((_, reject) => {
      timeoutId = setTimeout(() => {
        reject(new Error("Shopping cart update timeout"));
      }, timeout);
    });

    // 요청 페이로드 구성
    const payload = {
      product_id: productId,
      quantity,
    };

    // 사이즈 정보가 있으면 추가
    if (size) {
      payload.size = size;
      console.log(`Adding size (${size}) to cart update payload`);
    }

    // variant_id가 있으면 추가
    if (variant_id) {
      payload.variant_id = variant_id;
    }

    console.log(
      `[Shopping Cart Update] Request payload:`,
      JSON.stringify(payload, null, 2)
    );

    // 실제 API 요청
    const fetchPromise = api.post("/shopping-cart/add", payload, { signal });

    // Promise.race로 둘 중 먼저 완료되는 것 반환
    const response = await Promise.race([fetchPromise, timeoutPromise]);

    // 타임아웃 취소
    clearTimeout(timeoutId);

    return response.data;
  } catch (error) {
    // AbortError 또는 타임아웃은 특별 처리
    if (error.name === "AbortError" || error.message.includes("timeout")) {
      console.warn("Shopping cart update aborted:", error.message);
      return { success: true, local_only: true, aborted: true };
    }

    // 그 외 오류 - 로컬 처리만 하고 계속 진행
    console.error("Shopping cart update failed:", error.message);
    if (error.response) {
      console.error("Response status:", error.response.status);
      console.error(
        "Response data:",
        JSON.stringify(error.response.data, null, 2)
      );
    }
    return { success: true, local_only: true, error: error.message };
  }
};

// 장바구니 비우기 함수 - 최적화 및 타임아웃 지원
export const clearShoppingCart = async (options = {}) => {
  try {
    const { signal, timeout = 8000 } = options;

    // 타임아웃 설정
    let timeoutId;
    const timeoutPromise = new Promise((_, reject) => {
      timeoutId = setTimeout(() => {
        reject(new Error("Shopping cart clear timeout"));
      }, timeout);
    });

    // 장바구니 조회 (최대 2초 타임아웃)
    let cartItems = [];
    try {
      const cartResponse = await fetchShoppingCart({
        signal,
        timeout: Math.min(2000, timeout / 2),
      });
      cartItems = cartResponse.items || [];
      console.log("Current cart items:", JSON.stringify(cartItems));
    } catch (error) {
      // 장바구니 조회 실패해도 계속 진행 (빈 장바구니로 가정)
      console.warn("Shopping cart fetch failed:", error.message);
    }

    // 장바구니가 이미 비어있으면 성공 반환
    if (!cartItems || cartItems.length === 0) {
      clearTimeout(timeoutId);
      console.log("Cart is already empty, no need to clear");
      return { success: true };
    }

    console.log(`Found ${cartItems.length} items to remove from cart`);

    // 각 상품 제거 요청을 병렬로 처리 (각각 최대 2초 타임아웃)
    const removePromises = cartItems.map((item) =>
      removeCartItem(item.product_id, 0, {
        // 수량을 0으로 설정하여 제거 요청
        signal,
        timeout: Math.min(2000, timeout / 3),
        size: item.size,
      })
    );

    // 모든 상품 제거 요청의 결과 기다리기
    const results = await Promise.allSettled(removePromises);

    // 타임아웃 취소
    clearTimeout(timeoutId);

    // 성공 여부 계산 (일부만 성공해도 성공으로 간주)
    const successCount = results.filter(
      (r) => r.status === "fulfilled" && r.value?.success !== false
    ).length;

    console.log(
      `Successfully removed ${successCount} of ${cartItems.length} items`
    );

    return {
      success: true,
      removed: successCount,
      total: cartItems.length,
      partially: successCount < cartItems.length && successCount > 0,
    };
  } catch (error) {
    // AbortError 또는 타임아웃은 특별 처리
    if (error.name === "AbortError" || error.message.includes("timeout")) {
      console.warn("Shopping cart clear aborted:", error.message);
      return { success: true, aborted: true };
    }

    // 그 외 오류 - 성공으로 처리하고 계속 진행
    console.error("Shopping cart clear failed:", error.message);
    return { success: true, error: error.message };
  }
};

// 장바구니 완전 동기화 - 재시도 및 타임아웃 메커니즘 추가
export const syncShoppingCartWithLocal = async (
  localCartItems,
  options = {}
) => {
  const { signal, timeout = 10000, maxRetries = 2 } = options;

  // 상품이 없으면 즉시 성공 반환
  if (!localCartItems || localCartItems.length === 0) {
    return { success: true, syncedItems: 0, totalItems: 0 };
  }

  // 타임아웃 설정
  let timeoutId;
  const timeoutPromise = new Promise((_, reject) => {
    timeoutId = setTimeout(() => {
      reject(new Error("Shopping cart sync timeout"));
    }, timeout);
  });

  try {
    // 1. 장바구니 비우기 시도 (최대 3초)
    let clearSuccess = false;
    for (let attempt = 0; attempt < Math.min(2, maxRetries); attempt++) {
      try {
        const clearResult = await clearShoppingCart({
          signal,
          timeout: Math.min(3000, timeout / 3),
        });

        if (clearResult.success) {
          clearSuccess = true;
          break;
        }
      } catch (error) {
        console.warn(
          `Shopping cart clear attempt ${attempt + 1} failed:`,
          error.message
        );
        if (attempt === Math.min(2, maxRetries) - 1) {
          // 마지막 시도 실패 - 경고만 기록하고 계속 진행
        }
      }
    }

    // 중단 신호 확인
    if (signal?.aborted) {
      clearTimeout(timeoutId);
      return { success: false, aborted: true };
    }

    // 2. 로컬 상품 추가 처리 - 각 상품마다 최대 2번 재시도
    const results = [];
    const failedItems = [];

    // 상품별 타임아웃 계산 (전체 남은 시간을 상품 수로 분배)
    const itemStartTime = Date.now();
    const remainingTime = timeout - (Date.now() - itemStartTime);
    const itemTimeout = Math.max(
      1000,
      Math.min(2000, remainingTime / localCartItems.length)
    );

    for (const item of localCartItems) {
      if (!item || !item.product_id) continue;

      // 중단 신호 확인
      if (signal?.aborted) break;

      // 남은 시간 확인 - 타임아웃 가능성 있으면 중단
      if (Date.now() - itemStartTime > timeout * 0.9) {
        console.warn("Shopping cart sync timeout");
        break;
      }

      let itemSuccess = false;
      let lastError = null;

      // 각 상품마다 최대 재시도 횟수만큼 시도
      for (let attempt = 0; attempt < maxRetries; attempt++) {
        try {
          // 상품 정보 추출
          let productId = item.product_id;
          let size = item.size || null;

          console.log(
            `Syncing product ${productId}${size ? ` with size ${size}` : ""}`
          );

          // 상품 추가 요청 (타임아웃 포함) - 백엔드 API 스펙에 맞게 구성
          const payload = {
            product_id: productId,
            quantity: item.quantity,
          };

          // 사이즈 정보가 있으면 추가
          if (size) {
            payload.size = size;
          }

          console.log(`Payload for syncing: ${JSON.stringify(payload)}`);

          const addResult = await api.post("/shopping-cart/add", payload, {
            signal,
            timeout: itemTimeout,
            headers: { "Content-Type": "application/json" },
          });

          console.log(`Sync response: ${JSON.stringify(addResult.data)}`);

          results.push(addResult.data);
          itemSuccess = true;
          break;
        } catch (error) {
          lastError = error;
          console.warn(
            `상품 ${item.product_id} 추가 시도 ${attempt + 1} 실패:`,
            error.message
          );

          if (error.response) {
            console.warn(
              `Response status: ${
                error.response.status
              }, data: ${JSON.stringify(error.response.data)}`
            );
          }

          // 마지막 시도가 아니면 잠시 대기 후 재시도
          if (attempt < maxRetries - 1) {
            await new Promise((resolve) => setTimeout(resolve, 300));
          }
        }
      }

      // 모든 시도 실패한 상품 기록
      if (!itemSuccess) {
        failedItems.push({
          product_id: item.product_id,
          error: lastError?.message || "Unknown error",
        });
      }
    }

    // 타임아웃 취소
    clearTimeout(timeoutId);

    return {
      success: true,
      syncedItems: results.length,
      totalItems: localCartItems.length,
      failedItems: failedItems.length > 0 ? failedItems : undefined,
      clearSuccess,
    };
  } catch (error) {
    // 타임아웃 취소
    clearTimeout(timeoutId);

    // AbortError 또는 타임아웃은 특별 처리
    if (error.name === "AbortError" || error.message.includes("timeout")) {
      console.warn("Shopping cart sync aborted:", error.message);
      return {
        success: true,
        aborted: true,
        message:
          "Shopping cart sync aborted, but local cart is still available.",
      };
    }

    // 그 외 오류 - 로컬 유지하고 계속 진행
    console.error("Shopping cart sync failed:", error.message);
    return {
      success: true,
      error: error.message,
      message: "Shopping cart sync failed, but local cart is still available.",
    };
  }
};
