// app/api/endpoints/steps.js
import api from "@/app/lib/axios";
import dayjs from "dayjs";
import AsyncStorage from "@react-native-async-storage/async-storage";
import { Platform } from "react-native";

// API 호출 타임아웃 (15초에서 30초로 증가)
const API_TIMEOUT = 30000;

// 안드로이드에서 재시도 로직을 위한 유틸리티 함수
const retryWithTimeout = async (apiCall, maxRetries = 2) => {
  let lastError;

  for (let attempt = 0; attempt < maxRetries; attempt++) {
    try {
      return await apiCall();
    } catch (error) {
      lastError = error;
      console.warn(
        `API 호출 실패 (시도 ${attempt + 1}/${maxRetries}):`,
        error.message
      );

      // 네트워크 타임아웃이나 연결 오류일 경우에만 재시도
      if (
        error.code !== "ECONNABORTED" &&
        !error.message.includes("Network Error")
      ) {
        throw error; // 다른 오류는 바로 던짐
      }

      // 마지막 시도가 아니면 잠시 대기 후 재시도
      if (attempt < maxRetries - 1) {
        await new Promise((r) => setTimeout(r, 2000)); // 2초 대기
      }
    }
  }

  throw lastError; // 모든 재시도 실패 시 마지막 오류 던짐
};

export const getCurrentSteps = async () => {
  try {
    // 안드로이드에서는 재시도 로직 적용
    if (Platform.OS === "android") {
      return await retryWithTimeout(() =>
        api.get("/steps/current", { timeout: API_TIMEOUT * 2 })
      ).then((response) => response.data);
    } else {
      const response = await api.get("/steps/current", {
        timeout: API_TIMEOUT,
      });
      return response.data;
    }
  } catch (error) {
    console.warn("[API 오류] getCurrentSteps:", error.message);
    if (error.response?.status === 404) {
      throw error;
    }
    throw error;
  }
};

// Add validation to sendStepsToDB
export const sendStepsToDB = async (steps = 0) => {
  try {
    // Ensure steps is a non-negative integer
    if (!Number.isInteger(steps) || steps < 0) {
      console.warn("Invalid steps value:", steps);
      steps = Math.max(0, Math.floor(steps || 0));
    }

    // Add deduplication check using request_id
    const requestId = `${Date.now()}-${Math.floor(Math.random() * 1000000)}`;

    // Add rate limiting check
    const lastSyncTimeStr = await AsyncStorage.getItem("LAST_STEP_SYNC_TIME");
    const now = Date.now();
    if (lastSyncTimeStr && now - parseInt(lastSyncTimeStr) < 5000) {
      // 5 second cooldown
      console.log("Step sync too frequent, skipping");
      return null;
    }

    await AsyncStorage.setItem("LAST_STEP_SYNC_TIME", now.toString());

    // Calculate points using the standard formula: 1000 steps = 10 points
    const pointsEarned = Math.floor(steps / 1000) * 10;

    const payload = {
      record_date: dayjs().format("YYYY-MM-DD"),
      steps_count: steps,
      points_earned: pointsEarned,
      calories_burned: Number((steps * 0.04).toFixed(2)),
      distance_km: Number((steps * 0.0007).toFixed(2)),
      goal_achieved: false,
      sync_source: "mobile-app",
      points_calculation: {
        points_per_thousand: 10, // 1000 steps = 10 points
        multiplier: 1,
      },
      request_id: requestId, // Add unique request ID to help server deduplicate
      client_timestamp: Date.now(), // Add client timestamp for server validation
      platform: Platform.OS, // 플랫폼 정보 추가
    };

    console.log(
      `[${Platform.OS.toUpperCase()}] Sending new step record to server: ${steps} steps (Request ID: ${requestId})`
    );

    let response;
    // 안드로이드에서는 재시도 로직 적용
    if (Platform.OS === "android") {
      response = await retryWithTimeout(() =>
        api.post("/steps", payload, { timeout: API_TIMEOUT * 4 })
      );
    } else {
      // 타임아웃 값 증가 (APK 배포 환경에서 더 안정적인 통신을 위해)
      response = await api.post("/steps", payload, {
        timeout: API_TIMEOUT * 4, // 60초로 증가 (APK 배포 환경에서 더 안정적인 통신을 위해)
      });
    }

    console.log(
      `[${Platform.OS.toUpperCase()}] Step record creation successful. Server response:`,
      {
        status: response.status,
        stepsCount: response.data.steps_count,
        pointsEarned: response.data.points_earned,
        requestId: requestId,
      }
    );

    // 성공 시 로컬 저장
    try {
      // 안드로이드에서 로컬 스토리지 불일치 방지를 위해 사용자 ID 기반 키 추가
      const userId = await AsyncStorage.getItem("USER_ID");
      const dailyStepKey = userId ? `DAILY_STEP_${userId}` : "DAILY_STEP";
      const dailyDateKey = userId ? `DAILY_DATE_${userId}` : "DAILY_DATE";

      await AsyncStorage.setItem(
        dailyStepKey,
        String(response.data.steps_count)
      );
      await AsyncStorage.setItem(dailyDateKey, dayjs().format("YYYY-MM-DD"));

      // 기존 키도 유지 (하위 호환성)
      await AsyncStorage.setItem(
        "DAILY_STEP",
        String(response.data.steps_count)
      );
      await AsyncStorage.setItem("DAILY_DATE", dayjs().format("YYYY-MM-DD"));

      console.log(
        `[${Platform.OS.toUpperCase()}] 로컬 스토리지에 걸음수 저장 성공:`,
        response.data.steps_count
      );
    } catch (storageError) {
      console.warn(
        `[${Platform.OS.toUpperCase()}] 로컬 스토리지 저장 오류:`,
        storageError
      );
    }

    return response.data;
  } catch (error) {
    console.error(
      `[${Platform.OS.toUpperCase()}] Error sending steps to DB:`,
      error
    );

    // 안드로이드에서 오류 발생 시 로컬 캐시에 저장
    if (Platform.OS === "android") {
      try {
        const pendingStepsJSON =
          (await AsyncStorage.getItem("PENDING_STEPS_SYNC")) || "[]";
        const pendingSteps = JSON.parse(pendingStepsJSON);
        pendingSteps.push({
          steps,
          timestamp: Date.now(),
          date: dayjs().format("YYYY-MM-DD"),
        });
        // 최대 10개 항목만 유지
        const limitedPendingSteps = pendingSteps.slice(-10);
        await AsyncStorage.setItem(
          "PENDING_STEPS_SYNC",
          JSON.stringify(limitedPendingSteps)
        );
        console.log(
          "[ANDROID] 동기화 실패한 걸음수를 로컬에 임시 저장:",
          steps
        );
      } catch (cacheError) {
        console.error("[ANDROID] 실패한 걸음수 임시 저장 오류:", cacheError);
      }
    }

    throw error;
  }
};

export const accumulateSteps = async (stepsIncrement) => {
  try {
    if (!Number.isInteger(stepsIncrement) || stepsIncrement < 0) {
      console.warn("Invalid steps increment:", stepsIncrement);
      stepsIncrement = Math.max(0, Math.floor(stepsIncrement || 0));
    }

    // Skip zero increments to avoid unnecessary API calls
    if (stepsIncrement === 0) {
      console.log("Skipping API call for zero steps increment");
      // Return a mock response to avoid errors in calling code
      return { steps_count: 0, message: "No increment to sync" };
    }

    // Generate a unique request ID based on timestamp and random number to prevent duplicates
    const requestId = `${Date.now()}-${Math.floor(Math.random() * 1000000)}`;
    console.log(
      `[${Platform.OS.toUpperCase()}] Sending step increment to API: ${stepsIncrement} (Request ID: ${requestId})`
    );

    const payload = {
      steps_increment: stepsIncrement,
      points_calculation: {
        points_per_thousand: 10, // 1000 steps = 10 points
        multiplier: 1,
      },
      request_id: requestId, // Add unique request ID to help server deduplicate
      client_timestamp: Date.now(), // Add client timestamp for server validation
      platform: Platform.OS, // 플랫폼 정보 추가
    };

    let response;
    // 안드로이드에서는 재시도 로직 적용
    if (Platform.OS === "android") {
      response = await retryWithTimeout(() =>
        api.patch("/steps/accumulate", payload, { timeout: API_TIMEOUT * 4 })
      );
    } else {
      response = await api.patch("/steps/accumulate", payload, {
        timeout: API_TIMEOUT * 4, // 120초로 증가 (APK 환경용)
      });
    }

    console.log(
      `[${Platform.OS.toUpperCase()}] Step accumulation successful. Server response:`,
      {
        status: response.status,
        newTotal: response.data.steps_count,
        increment: stepsIncrement,
        requestId: requestId,
      }
    );

    // 성공 시 로컬 저장
    try {
      // 안드로이드에서 로컬 스토리지 불일치 방지를 위해 사용자 ID 기반 키 추가
      const userId = await AsyncStorage.getItem("USER_ID");
      const dailyStepKey = userId ? `DAILY_STEP_${userId}` : "DAILY_STEP";

      await AsyncStorage.setItem(
        dailyStepKey,
        String(response.data.steps_count)
      );

      // 기존 키도 유지 (하위 호환성)
      await AsyncStorage.setItem(
        "DAILY_STEP",
        String(response.data.steps_count)
      );

      console.log(
        `[${Platform.OS.toUpperCase()}] 로컬 스토리지에 걸음수 저장 성공:`,
        response.data.steps_count
      );
    } catch (storageError) {
      console.warn(
        `[${Platform.OS.toUpperCase()}] 로컬 스토리지 저장 오류:`,
        storageError
      );
    }

    return response.data;
  } catch (error) {
    if (error.code === "ECONNABORTED") {
      console.warn(
        `[${Platform.OS.toUpperCase()}] [네트워크 타임아웃] /steps/accumulate:`,
        error.message
      );
    } else {
      console.warn(
        `[${Platform.OS.toUpperCase()}] [API 오류] accumulateSteps:`,
        error.message
      );
      // Log more details about the error for debugging
      if (error.response) {
        console.warn(`Server responded with status ${error.response.status}`);
        console.warn("Response data:", error.response.data);
      } else if (error.request) {
        console.warn("Request made but no response received");
      } else {
        console.warn("Error configuring request:", error.message);
      }
    }

    // 안드로이드에서 오류 발생 시 로컬 캐시에 저장
    if (Platform.OS === "android") {
      try {
        const pendingIncrementsJSON =
          (await AsyncStorage.getItem("PENDING_STEPS_INCREMENTS")) || "[]";
        const pendingIncrements = JSON.parse(pendingIncrementsJSON);
        pendingIncrements.push({
          increment: stepsIncrement,
          timestamp: Date.now(),
          date: dayjs().format("YYYY-MM-DD"),
        });
        // 최대 10개 항목만 유지
        const limitedPendingIncrements = pendingIncrements.slice(-10);
        await AsyncStorage.setItem(
          "PENDING_STEPS_INCREMENTS",
          JSON.stringify(limitedPendingIncrements)
        );
        console.log(
          "[ANDROID] 동기화 실패한 걸음수 증가분을 로컬에 임시 저장:",
          stepsIncrement
        );
      } catch (cacheError) {
        console.error(
          "[ANDROID] 실패한 걸음수 증가분 임시 저장 오류:",
          cacheError
        );
      }
    }

    throw error;
  }
};

export const getAllSteps = async () => {
  try {
    // 안드로이드에서는 재시도 로직 적용
    if (Platform.OS === "android") {
      return await retryWithTimeout(() =>
        api.get("/steps/all", { timeout: API_TIMEOUT * 3 })
      ).then((response) => response.data);
    } else {
      const response = await api.get("/steps/all", {
        timeout: API_TIMEOUT * 2, // 60초로 증가 (APK 환경용)
      });
      return response.data;
    }
  } catch (error) {
    if (error.code === "ECONNABORTED") {
      console.warn(
        `[${Platform.OS.toUpperCase()}] [네트워크 타임아웃] /steps/all:`,
        error.message
      );
    } else {
      console.warn(
        `[${Platform.OS.toUpperCase()}] [API 오류] getAllSteps:`,
        error.message
      );
    }
    throw error;
  }
};

// 안드로이드용 보류 중인 걸음수 동기화 함수 추가
export const syncPendingSteps = async () => {
  if (Platform.OS !== "android") return;

  try {
    // 보류 중인 걸음수 로드
    const pendingStepsJSON =
      (await AsyncStorage.getItem("PENDING_STEPS_SYNC")) || "[]";
    const pendingSteps = JSON.parse(pendingStepsJSON);

    if (pendingSteps.length === 0) return;

    console.log(
      "[ANDROID] 보류 중인 걸음수 동기화 시도:",
      pendingSteps.length,
      "개 항목"
    );

    // 가장 최근 항목만 시도 (이전 값은 무시)
    const latestPending = pendingSteps[pendingSteps.length - 1];

    // 걸음수 동기화 시도
    await sendStepsToDB(latestPending.steps);

    // 성공 시 보류 목록 비우기
    await AsyncStorage.setItem("PENDING_STEPS_SYNC", "[]");
    console.log("[ANDROID] 보류 중인 걸음수 동기화 성공");
  } catch (error) {
    console.error("[ANDROID] 보류 중인 걸음수 동기화 실패:", error);
  }

  // 증가분 동기화 시도
  try {
    const pendingIncrementsJSON =
      (await AsyncStorage.getItem("PENDING_STEPS_INCREMENTS")) || "[]";
    const pendingIncrements = JSON.parse(pendingIncrementsJSON);

    if (pendingIncrements.length === 0) return;

    console.log(
      "[ANDROID] 보류 중인 걸음수 증가분 동기화 시도:",
      pendingIncrements.length,
      "개 항목"
    );

    // 가장 최근 항목만 시도
    const latestIncrement = pendingIncrements[pendingIncrements.length - 1];

    // 증가분 동기화 시도
    await accumulateSteps(latestIncrement.increment);

    // 성공 시 보류 목록 비우기
    await AsyncStorage.setItem("PENDING_STEPS_INCREMENTS", "[]");
    console.log("[ANDROID] 보류 중인 걸음수 증가분 동기화 성공");
  } catch (incrementError) {
    console.error(
      "[ANDROID] 보류 중인 걸음수 증가분 동기화 실패:",
      incrementError
    );
  }
};
