import React, { useEffect, useRef, useState } from "react";
import {
  View,
  StyleSheet,
  Dimensions,
  TouchableOpacity,
  Text,
} from "react-native";
import Svg, { G, Path, Text as SvgText } from "react-native-svg";
import Animated, {
  useSharedValue,
  useAnimatedStyle,
  withTiming,
  Easing,
  runOnJS,
} from "react-native-reanimated";
import { Colors } from "@/assets/styles/App.styles";

const { width } = Dimensions.get("window");
const WHEEL_SIZE = width - 80;
const WHEEL_CENTER = WHEEL_SIZE / 2;
const FONT_SIZE = 18;
const MULTIPLIERS = ["1.2", "1.5", "1.7", "2", "2.5", "3"];
const COLORS = [
  Colors.primary,
  Colors.neutral,
  Colors.accent,
  "#FF6B6B",
  "#4ECDC4",
  "#45B7D1",
];
interface WheelOfFortuneProps {
  onSpinComplete: (multiplier: string) => void;
}

const WheelOfFortune: React.FC<WheelOfFortuneProps> = ({ onSpinComplete }) => {
  const [result, setResult] = useState<string | null>(null);
  const rotation = useSharedValue(0);
  const isSpinning = useRef(false);

  const wheelStyle = useAnimatedStyle(() => {
    return {
      transform: [{ rotate: `${rotation.value}deg` }],
    };
  });

  const createWheelPaths = () => {
    const angleSlice = 360 / MULTIPLIERS.length;
    const paths = MULTIPLIERS.map((_, index) => {
      const startAngle = index * angleSlice;
      const endAngle = (index + 1) * angleSlice;
      const x1 =
        WHEEL_CENTER + WHEEL_CENTER * Math.cos((Math.PI * startAngle) / 180);
      const y1 =
        WHEEL_CENTER + WHEEL_CENTER * Math.sin((Math.PI * startAngle) / 180);
      const x2 =
        WHEEL_CENTER + WHEEL_CENTER * Math.cos((Math.PI * endAngle) / 180);
      const y2 =
        WHEEL_CENTER + WHEEL_CENTER * Math.sin((Math.PI * endAngle) / 180);

      return `M${WHEEL_CENTER},${WHEEL_CENTER} L${x1},${y1} A${WHEEL_CENTER},${WHEEL_CENTER} 0 0,1 ${x2},${y2} Z`;
    });

    return paths;
  };

  const spinWheel = () => {
    if (isSpinning.current) return;

    isSpinning.current = true;
    setResult(null);

    // Reset rotation to 0 before starting new spin
    rotation.value = 0;

    const randomDegrees = Math.floor(Math.random() * 360) + 1440; // At least 4 full rotations
    rotation.value = withTiming(
      randomDegrees,
      {
        duration: 5000,
        easing: Easing.out(Easing.cubic),
      },
      (finished) => {
        if (finished) {
          runOnJS(handleSpinComplete)(randomDegrees);
        }
      }
    );
  };

  //Working with 6 length MULTIPLIERS array
  const handleSpinComplete = (finalRotation: number) => {
    const normalizedRotation = finalRotation % 360;
    const sliceAngle = 360 / MULTIPLIERS.length;
    const multiplierIndex = Math.floor(
      ((360 - normalizedRotation + sliceAngle / 2) % 360) / sliceAngle
    );
    // Use modulo to wrap around the array safely
    const safeIndex =
      (multiplierIndex - 2 + MULTIPLIERS.length) % MULTIPLIERS.length;
    const selectedMultiplier = MULTIPLIERS[safeIndex];
    setResult(selectedMultiplier);
    onSpinComplete(selectedMultiplier);
    isSpinning.current = false;
  };

  return (
    <View style={styles.container}>
      <View style={styles.wheelContainer}>
        <View style={styles.pointer}>
          <View style={styles.InnerSmallPointer}></View>
        </View>
        <Animated.View style={[styles.wheel, wheelStyle]}>
          <Svg
            width={WHEEL_SIZE}
            height={WHEEL_SIZE}
            viewBox={`0 0 ${WHEEL_SIZE} ${WHEEL_SIZE}`}
          >
            <G>
              {createWheelPaths().map((path, index) => (
                <Path
                  key={index}
                  d={path}
                  fill={COLORS[index % COLORS.length]}
                />
              ))}
              {MULTIPLIERS.map((multiplier, index) => {
                const angle =
                  ((360 / MULTIPLIERS.length) * index +
                    180 / MULTIPLIERS.length) *
                  (Math.PI / 180);
                const x = WHEEL_CENTER + (WHEEL_CENTER - 40) * Math.cos(angle);
                const y = WHEEL_CENTER + (WHEEL_CENTER - 40) * Math.sin(angle);
                return (
                  <SvgText
                    key={index}
                    x={x}
                    y={y}
                    fill={Colors.white}
                    fontSize={FONT_SIZE}
                    fontWeight="bold"
                    textAnchor="middle"
                    alignmentBaseline="middle"
                  >
                    {multiplier}x
                  </SvgText>
                );
              })}
            </G>
          </Svg>
        </Animated.View>
      </View>
      <TouchableOpacity style={styles.button} onPress={spinWheel}>
        <Text style={styles.buttonText}>Çarkı Çevir</Text>
      </TouchableOpacity>
      {result && (
        <View style={styles.resultContainer}>
          <Text style={styles.resultText}>Seçilen Çarpan: {result}x</Text>
        </View>
      )}
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    alignItems: "center",
    justifyContent: "center",
  },
  wheelContainer: {
    width: WHEEL_SIZE,
    height: WHEEL_SIZE,
    alignItems: "center",
    justifyContent: "center",
  },
  wheel: {
    width: WHEEL_SIZE,
    height: WHEEL_SIZE,
    alignItems: "center",
    justifyContent: "center",
  },
  pointer: {
    width: 0,
    height: 0,
    backgroundColor: "transparent",
    borderStyle: "solid",
    borderLeftWidth: 25,
    borderRightWidth: 25,
    borderTopWidth: 45,
    borderLeftColor: "transparent",
    borderRightColor: "transparent",
    borderTopColor: Colors.primary,
    position: "absolute",
    top: -25,
    zIndex: 1,
    alignItems: "center",
    justifyContent: "center",
  },
  InnerSmallPointer: {
    width: 0,
    height: 0,
    backgroundColor: "transparent",
    borderStyle: "solid",
    borderLeftWidth: 17,
    borderRightWidth: 17,
    borderTopWidth: 30,
    borderLeftColor: "transparent",
    borderRightColor: "transparent",
    borderTopColor: Colors.white,
    position: "absolute",
    zIndex: 1,
    top: -40,
  },

  button: {
    marginTop: 20,
    backgroundColor: Colors.primary,
    paddingHorizontal: 20,
    paddingVertical: 10,
    borderRadius: 10,
  },
  buttonText: {
    color: Colors.white,
    fontSize: 18,
    fontFamily: "montbold",
  },
  resultContainer: {
    marginTop: 20,
    padding: 10,
    backgroundColor: Colors.white,
    borderRadius: 10,
    shadowColor: Colors.tertiary,
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  resultText: {
    fontSize: 18,
    fontFamily: "montbold",
    color: Colors.primary,
  },
});

export default WheelOfFortune;
