import React, { useEffect } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  Image,
  LayoutAnimation,
  ActivityIndicator,
} from "react-native";
import { useNavigation } from "expo-router";
import { Feather, Ionicons } from "@expo/vector-icons";
import { Colors, styles } from "@/assets/styles/App.styles";
import Topbar from "@/app/components/Topbar";
import Toast from "@/app/components/common/ToastMessage";
import Toolbar from "@/app/components/Toolbar";
import useCartStore from "@/app/store/cartStore";

// The backend response uses the "stock" key, so here we use the already stored stockCount value internally.
// 실제 API 호출을 모방하는 함수
async function fetchUpdatedStocksForCartItems(cartItems) {
  try {
    if (!cartItems || cartItems.length === 0) {
      return {};
    }

    // 백엔드 API 호출 대신 로컬에서 처리
    const updatedStocks = {};
    cartItems.forEach((item) => {
      if (item && item.product_id) {
        // 현재 stockCount 값을 그대로 사용
        updatedStocks[item.product_id] = item.stockCount || 0;
      }
    });

    // 인위적인 지연 제거
    return updatedStocks;
  } catch (error) {
    console.error("Error in fetchUpdatedStocksForCartItems:", error);
    // 오류 발생 시 빈 객체 반환하여 UI가 정상 작동하도록 함
    return {};
  }
}

const Cart = () => {
  const navigation = useNavigation();
  const [isLoading, setIsLoading] = React.useState(false);
  const [isMounted, setIsMounted] = React.useState(true);
  const loadingRef = React.useRef(false); // Add this line

  // Get the latest cart state
  const cartItems = useCartStore((state) => state.items);
  const updateItemQuantity = useCartStore((state) => state.updateItemQuantity);
  const removeItem = useCartStore((state) => state.removeItem);
  const syncCartStocks = useCartStore((state) => state.syncCartStocks);

  // 재고 동기화 함수
  const syncStocks = React.useCallback(async () => {
    if (!isMounted || loadingRef.current) return;

    try {
      loadingRef.current = true;
      setIsLoading(true);

      // 백엔드 API에 요청하는 대신 장바구니 항목의 stockCount 값만 사용
      const updatedStocks = await fetchUpdatedStocksForCartItems(cartItems);

      if (isMounted) {
        syncCartStocks(updatedStocks);

        // 재고 변경 알림 (백엔드와 실제 동기화 시에만 의미가 있음)
        // 현재는 로컬 값만 사용하므로 생략 가능
      }
    } catch (error) {
      console.error("Error occurred during stock synchronization:", error);
      // 오류가 발생해도 UI가 정상 작동하도록 함
    } finally {
      if (isMounted) {
        setIsLoading(false);
        loadingRef.current = false;

        // 로딩 시간이 너무 짧으면 깜빡임이 발생할 수 있으므로,
        // 최소 로딩 시간 보장 (필요 시)
        // setTimeout(() => {
        //   if (isMounted) {
        //     setIsLoading(false);
        //     loadingRef.current = false;
        //   }
        // }, 300);
      }
    }
  }, [cartItems, syncCartStocks, isMounted]);

  // 컴포넌트 마운트/언마운트 추적
  React.useEffect(() => {
    setIsMounted(true);
    syncStocks(); // Move initial sync here

    return () => {
      setIsMounted(false);
      loadingRef.current = false; // Reset loading ref on unmount
    };
  }, []); // Remove syncStocks from dependency array

  // Remove this useEffect as we moved the sync call to the mount effect
  // React.useEffect(() => {
  //   syncStocks();
  // }, [syncStocks]);

  const handleRemoveItem = (productId) => {
    if (!productId) return;

    try {
      removeItem(productId);
      Toast("Ürün sepetten kaldırıldı", "", "success");
      LayoutAnimation.configureNext(LayoutAnimation.Presets.easeInEaseOut);
    } catch (error) {
      console.error("Error removing item:", error);
    }
  };

  const handleQuantityChange = (productId, increment) => {
    if (!productId) return;

    try {
      const currentItem = cartItems.find(
        (item) => item.product_id === productId
      );
      if (!currentItem) return;

      const newQuantity = currentItem.quantity + increment;
      if (newQuantity < 1 || newQuantity > currentItem.stockCount) return;

      updateItemQuantity(productId, newQuantity);
    } catch (error) {
      console.error("Error updating quantity:", error);
    }
  };

  const handleContinueShopping = () => {
    try {
      // 페이지 전환 전에 상태 초기화 작업이 필요하다면 여기서 수행
      navigation.navigate("pages/shop/shop");
    } catch (error) {
      console.error("Navigation error:", error);
    }
  };

  const calculateTotal = () =>
    cartItems.reduce(
      (total, item) => total + item.points_required * item.quantity,
      0
    );

  const renderCartItem = (item, index) => (
    <View
      key={item.product_id ? `${item.product_id}-${index}` : index}
      style={styles.cartItemContainer}
    >
      <View style={styles.cartItemImageWrapper}>
        <Image
          source={{ uri: item.thumbnail }}
          style={[styles.cartItemImage, { resizeMode: "contain" }]}
        />
        <View style={styles.cartItemBadge}>
          <Text style={styles.cartItemBadgeText}>{item.quantity}</Text>
        </View>
      </View>
      <View style={styles.cartItemContent}>
        <View style={styles.cartItemHeader}>
          <View style={{ flex: 1, marginRight: 8 }}>
            <Text
              style={styles.cartItemName}
              numberOfLines={1}
              ellipsizeMode="tail"
            >
              {item.name}
            </Text>
            <Text style={styles.cartItemPrice}>
              {item.points_required} Adım Puan
            </Text>
          </View>
          <TouchableOpacity
            onPress={() => handleRemoveItem(item.product_id)}
            style={styles.cartItemRemove}
          >
            <Feather name="x-circle" size={24} color={Colors.danger} />
          </TouchableOpacity>
        </View>
        <View style={styles.cartItemFooter}>
          <View style={styles.cartItemQuantityControls}>
            <TouchableOpacity
              onPress={() => handleQuantityChange(item.product_id, -1)}
              disabled={item.quantity === 1}
              style={[
                styles.quantityButton,
                item.quantity === 1 && { opacity: 0.5 },
              ]}
            >
              <Feather name="minus" size={16} color={Colors.primary} />
            </TouchableOpacity>
            <Text style={styles.quantityText}>{item.quantity}</Text>
            <TouchableOpacity
              onPress={() => handleQuantityChange(item.product_id, 1)}
              disabled={item.quantity >= item.stockCount}
              style={[
                styles.quantityButton,
                item.quantity >= item.stockCount && { opacity: 0.5 },
              ]}
            >
              <Feather name="plus" size={16} color={Colors.primary} />
            </TouchableOpacity>
          </View>
          <Text style={styles.cartItemTotal}>
            {item.points_required * item.quantity} Adım Puan
          </Text>
        </View>
      </View>
    </View>
  );

  return (
    <View style={styles.MainContainer}>
      <Topbar title="Sepetim" />
      <ScrollView
        style={{ width: "100%" }}
        showsVerticalScrollIndicator={false}
      >
        <View style={styles.cartContainer}>
          {isLoading && cartItems.length === 0 ? (
            <View style={styles.loadingContainer}>
              <ActivityIndicator size="large" color={Colors.primary} />
            </View>
          ) : cartItems.length > 0 ? (
            <>
              {cartItems.map((item, index) =>
                item ? renderCartItem(item, index) : null
              )}
              <View style={styles.cartSummary}>
                <View style={styles.cartSummaryRow}>
                  <Text style={styles.cartSummaryLabel}>Ara Toplam</Text>
                  <Text style={styles.cartSummaryValue}>
                    {calculateTotal()} Adım Puan
                  </Text>
                </View>
                <View style={styles.cartSummaryRow}>
                  <Text style={styles.cartSummaryLabel}>Kargo</Text>
                  <Text style={styles.cartSummaryValue}>Ücretsiz</Text>
                </View>
                <View style={[styles.cartSummaryRow, styles.cartTotalRow]}>
                  <Text style={styles.cartTotalLabel}>Toplam</Text>
                  <Text style={styles.cartTotalValue}>
                    {calculateTotal()} Adım Puan
                  </Text>
                </View>
              </View>
              <TouchableOpacity
                style={styles.checkoutButton}
                onPress={() => {
                  // Verify final stock/points before attempting payment
                  navigation.navigate("components/checkout");
                }}
              >
                <Ionicons name="card-outline" size={24} color={Colors.white} />
                <Text style={styles.checkoutButtonText}>Ödemeye Geç</Text>
              </TouchableOpacity>
            </>
          ) : (
            <View style={styles.emptyCartContainer}>
              <Feather name="shopping-cart" size={64} color={Colors.tertiary} />
              <Text style={styles.emptyCartText}>Sepetiniz boş</Text>
              <TouchableOpacity
                style={styles.continueShopping}
                onPress={handleContinueShopping}
              >
                <Text style={styles.continueShoppingText}>
                  Alışverişe Devam Et
                </Text>
              </TouchableOpacity>
            </View>
          )}
        </View>
      </ScrollView>
      <Toolbar />
    </View>
  );
};

export default Cart;
