import React, { useState, useEffect } from "react";
import { View, ScrollView } from "react-native";
import { useNavigation } from "expo-router";
import { Colors, styles } from "@/assets/styles/App.styles";
import Topbar from "@/app/components/Topbar";
import Toast from "@/app/components/common/ToastMessage";
import Toolbar from "@/app/components/Toolbar";
import useCartStore from "@/app/store/cartStore";
import { useAddresses } from "@/app/hooks/queries/useAddresses";
import { useCheckout } from "@/app/hooks/mutations/useCheckout";
import { useUserPoints } from "@/app/hooks/queries/useUserPoints";
import useCartSync from "@/app/hooks/queries/useCartSync";

// 컴포넌트 분리
import AddressSection from "@/app/components/checkout/AddressSection";
import OrderSummary from "@/app/components/checkout/OrderSummary";

const Checkout = () => {
  const navigation = useNavigation();
  const [selectedAddress, setSelectedAddress] = useState(null);
  const [newAddressMode, setNewAddressMode] = useState(false);
  const [formData, setFormData] = useState({
    addressTitle: "",
    fullName: "",
    phone: "",
    city: "",
    fullAddress: "",
    postalCode: "",
    saveAddress: true,
  });

  // Get cart items from cart store
  const cartItems = useCartStore((state) => state.items);

  // 사용자 포인트 조회
  const {
    data: pointsData,
    refetch: refetchPoints,
    isLoading: isLoadingPoints,
  } = useUserPoints();

  // Fetch addresses from API
  const {
    data: addresses,
    isLoading: isLoadingAddresses,
    error: addressError,
  } = useAddresses();

  // Custom hook for cart synchronization
  const {
    isSyncingCart,
    isCheckingPoints,
    syncCartWithBackend,
    checkUserPoints,
    clearCart,
  } = useCartSync();

  // Checkout mutation
  const { mutate: checkout, isLoading: isCheckingOut } = useCheckout();

  // 페이지 진입시 항상 포인트 정보 갱신
  useEffect(() => {
    refetchPoints();
  }, [refetchPoints]);

  // Debug addresses data
  useEffect(() => {
    if (addresses) {
      console.log("Loaded addresses:", JSON.stringify(addresses));
    }
    if (addressError) {
      console.error("Error loading addresses:", addressError);
    }
  }, [addresses, addressError]);

  // Debug cart items - 한 번만 로깅하도록 조건 추가
  useEffect(() => {
    // 컴포넌트 마운트시 한 번만 로깅
    const cartItemCount = cartItems.length;
    console.log(
      `Cart has ${cartItemCount} items for checkout:`,
      JSON.stringify(
        cartItems.map(
          (item) => `${item.name} (${item.product_id}) x${item.quantity}`
        )
      )
    );
  }, []); // 의존성 배열을 비워 마운트시 한 번만 실행되도록 함

  // 컴포넌트 마운트시 자동으로 장바구니 동기화
  useEffect(() => {
    if (cartItems.length > 0) {
      console.log("Automatically syncing cart on mount...");
      syncCartWithBackend();
    }
  }, []); // 의존성 배열을 비워 마운트시 한 번만 실행되도록 함

  // Force new address mode if no addresses
  useEffect(() => {
    if (addresses && addresses.length === 0) {
      console.log("No saved addresses, forcing new address mode");
      setNewAddressMode(true);
    }
  }, [addresses]);

  // Set selected address from fetched addresses if available
  useEffect(() => {
    if (
      addresses &&
      addresses.length > 0 &&
      !selectedAddress &&
      !newAddressMode
    ) {
      console.log("Setting default address");
      const defaultAddress =
        addresses.find((addr) => addr.is_default) || addresses[0];
      setSelectedAddress(defaultAddress);
    }
  }, [addresses, selectedAddress, newAddressMode]);

  // For debugging: Log state changes
  useEffect(() => {
    console.log("newAddressMode:", newAddressMode);
    console.log("selectedAddress:", selectedAddress);
  }, [newAddressMode, selectedAddress]);

  // 포인트 정보 로깅
  useEffect(() => {
    if (pointsData) {
      console.log("Current user points:", pointsData.total_points);
    }
  }, [pointsData]);

  const handleSubmit = async () => {
    // 제출 전 포인트 정보 최신화
    await refetchPoints();

    // 제출 전 장바구니 확인
    console.log(
      "Preparing to submit order with cart items:",
      cartItems.map(
        (item) => `${item.name} (${item.product_id}) x${item.quantity}`
      )
    );

    if (!cartItems || cartItems.length === 0) {
      Toast("Hata", "Sepetiniz boş. Lütfen ürün ekleyin.", "danger");
      return;
    }

    // 체크아웃 전 사용자 포인트 확인
    console.log("Checking user points before checkout...");
    const pointsCheck = await checkUserPoints();
    if (!pointsCheck.success || !pointsCheck.hasEnoughPoints) {
      // 포인트 부족 시 메시지는 checkUserPoints 함수 내에서 표시됨
      return;
    }

    // 체크아웃 전 항상 장바구니 동기화
    console.log("Syncing cart before checkout...");
    const syncSuccess = await syncCartWithBackend();
    if (!syncSuccess) {
      Toast("Hata", "Sepet güncellenemedi. Lütfen tekrar deneyin.", "danger");
      return;
    }

    let checkoutData;

    if (!newAddressMode && selectedAddress) {
      // Using a saved address
      checkoutData = {
        shipping_details: {
          name: selectedAddress.recipient_name,
          address: selectedAddress.address_line1,
          city: selectedAddress.city,
          postal_code: selectedAddress.postal_code || "00",
          country: selectedAddress.country,
          phone: selectedAddress.phone,
        },
        save_address: false,
        address_title: selectedAddress.title,
      };
    } else {
      // Using a new address
      if (
        !formData.fullName ||
        !formData.phone ||
        !formData.fullAddress ||
        !formData.city
      ) {
        Toast("Hata", "Lütfen tüm alanları doldurun", "danger");
        return;
      }

      checkoutData = {
        shipping_details: {
          name: formData.fullName,
          address: formData.fullAddress,
          city: formData.city,
          postal_code: formData.postalCode || "00",
          country: "TR",
          phone: formData.phone,
        },
        save_address: formData.saveAddress,
        address_title: formData.addressTitle || "Ev",
      };
    }

    // Note: Removed cart_items from checkoutData since backend uses its own cart
    console.log("Submitting checkout data:", JSON.stringify(checkoutData));

    // Execute checkout mutation
    checkout(checkoutData, {
      onSuccess: (data) => {
        // 주문 성공 후 최신 포인트 정보 가져오기
        refetchPoints();

        // 주문 성공 후 장바구니 비우기
        clearCart()
          .then(() => {
            Toast(
              "Siparişiniz alındı",
              "Ödemeniz başarıyla gerçekleşti",
              "success"
            );
            navigation.reset({
              index: 0,
              routes: [{ name: "pages/home/homepage" }],
            });
          })
          .catch((error) => {
            console.error("Error clearing cart after order:", error);
            // 에러가 발생해도 홈페이지로 이동
            Toast(
              "Siparişiniz alındı",
              "Ödemeniz başarıyla gerçekleşti",
              "success"
            );
            navigation.reset({
              index: 0,
              routes: [{ name: "pages/home/homepage" }],
            });
          });
      },
      onError: (error) => {
        console.error("Checkout error details:", error);

        // 포인트 부족 오류인지 확인
        if (
          error.response?.data?.detail &&
          error.response.data.detail.includes("Not enough points")
        ) {
          const errorMsg = error.response.data.detail;
          const match = errorMsg.match(/Required: (\d+), Available: (\d+)/);

          if (match) {
            const required = match[1];
            const available = match[2];
            Toast(
              "Yetersiz Puan",
              `Bu siparişi tamamlamak için yeterli puanınız yok. Gerekli: ${required}, Mevcut: ${available}`,
              "warning"
            );
          } else {
            Toast(
              "Yetersiz Puan",
              "Bu siparişi tamamlamak için yeterli puanınız yok.",
              "warning"
            );
          }

          // 포인트 정보 갱신
          refetchPoints();
        } else {
          // 다른 오류 처리
          Toast(
            "Hata",
            error.response?.data?.detail ||
              error.message ||
              "Ödeme işlemi sırasında bir hata oluştu",
            "danger"
          );
        }
      },
    });
  };

  return (
    <View
      style={[styles.MainContainer, { backgroundColor: Colors.background }]}
    >
      <Topbar title="Ödeme" />

      <View style={{ flex: 1 }}>
        <ScrollView showsVerticalScrollIndicator={false}>
          <View style={styles.checkoutContainer}>
            {/* Address Section Component */}
            <AddressSection
              addresses={addresses}
              isLoadingAddresses={isLoadingAddresses}
              selectedAddress={selectedAddress}
              setSelectedAddress={setSelectedAddress}
              newAddressMode={newAddressMode}
              setNewAddressMode={setNewAddressMode}
              formData={formData}
              setFormData={setFormData}
            />

            {/* Order Summary Component */}
            <OrderSummary
              cartItems={cartItems}
              isCheckingOut={isCheckingOut}
              isSyncingCart={isSyncingCart}
              selectedAddress={selectedAddress}
              newAddressMode={newAddressMode}
              formData={formData}
              handleSubmit={handleSubmit}
            />
          </View>
        </ScrollView>
      </View>

      <Toolbar />
    </View>
  );
};

export default Checkout;
