import React from "react";
import { View, Text, TouchableOpacity, ActivityIndicator } from "react-native";
import { Feather } from "@expo/vector-icons";
import { Colors, styles } from "@/assets/styles/App.styles";
import { useUserPoints } from "@/app/hooks/queries/useUserPoints";

const OrderSummary = ({
  cartItems,
  isCheckingOut,
  isSyncingCart,
  selectedAddress,
  newAddressMode,
  formData,
  handleSubmit,
}) => {
  // useUserPoints 훅을 사용하여 포인트 정보 가져오기
  const {
    data: pointsData,
    isLoading: isLoadingPoints,
    error: pointsError,
  } = useUserPoints();

  // 사용자 포인트 (데이터가 없을 경우 0으로 기본값 설정)
  const userPoints = pointsData?.total_points || 0;

  // Calculate total points required
  const calculateTotal = () =>
    cartItems.reduce(
      (total, item) => total + item.points_required * item.quantity,
      0
    );

  // 총 필요 포인트
  const totalPoints = calculateTotal();
  // 포인트 부족 여부
  const hasEnoughPoints = userPoints >= totalPoints;

  // 로깅 추가
  React.useEffect(() => {
    if (pointsData) {
      console.log("User points from useUserPoints:", userPoints);
      console.log("Total required points:", totalPoints);
      console.log("Has enough points:", hasEnoughPoints);
    }
    if (pointsError) {
      console.error("Error loading user points:", pointsError);
    }
  }, [pointsData, pointsError, totalPoints, hasEnoughPoints, userPoints]);

  // Helper function to render product name with size if available
  const renderProductName = (item) => {
    const baseName = `${item.name} x ${item.quantity}`;
    if (item.size) {
      return (
        <View>
          <Text style={styles.cartItemSummaryName}>{baseName}</Text>
          <Text style={styles.cartItemSummarySize}>Beden: {item.size}</Text>
        </View>
      );
    }
    return <Text style={styles.cartItemSummaryName}>{baseName}</Text>;
  };

  return (
    <View style={[styles.checkoutSection, { marginBottom: 20 }]}>
      <View style={styles.checkoutSectionHeader}>
        <Feather name="shopping-bag" size={24} color={Colors.primary} />
        <Text style={styles.checkoutSectionTitle}>Sipariş Özeti</Text>
      </View>

      {/* Cart Items Summary */}
      {cartItems.length > 0 ? (
        <View style={styles.cartItemsSummary}>
          {cartItems.map((item) => (
            <View
              key={item.variant_id || item.product_id}
              style={styles.cartItemSummaryRow}
            >
              {renderProductName(item)}
              <Text style={styles.cartItemSummaryPrice}>
                {item.points_required * item.quantity} Adım Puan
              </Text>
            </View>
          ))}
        </View>
      ) : (
        <Text style={styles.emptyCartText}>
          Sepetinizde ürün bulunmamaktadır.
        </Text>
      )}

      {/* Total Summary */}
      <View style={styles.orderSummary}>
        <View style={styles.orderSummaryRow}>
          <Text style={styles.orderSummaryLabel}>Ara Toplam</Text>
          <Text style={styles.orderSummaryValue}>
            {calculateTotal()} Adım Puan
          </Text>
        </View>
        <View style={styles.orderSummaryRow}>
          <Text style={styles.orderSummaryLabel}>Kargo</Text>
          <Text style={styles.orderSummaryValue}>Ücretsiz</Text>
        </View>
        <View style={[styles.orderSummaryRow, styles.orderTotalRow]}>
          <Text style={styles.orderTotalLabel}>Toplam</Text>
          <Text style={styles.orderTotalValue}>
            {calculateTotal()} Adım Puan
          </Text>
        </View>

        {/* User Points Information */}
        <View
          style={[
            styles.orderSummaryRow,
            {
              marginTop: 10,
              paddingTop: 10,
              borderTopWidth: 1,
              borderTopColor: "#eee",
            },
          ]}
        >
          <Text style={styles.orderSummaryLabel}>Kullanılabilir Puanınız</Text>
          {isLoadingPoints ? (
            <ActivityIndicator size="small" color={Colors.primary} />
          ) : (
            <Text
              style={[
                styles.orderSummaryValue,
                !hasEnoughPoints ? { color: Colors.danger } : {},
              ]}
            >
              {userPoints} Adım Puan
            </Text>
          )}
        </View>

        {/* Points Status */}
        {!isLoadingPoints && (
          <View
            style={{
              marginTop: 5,
              padding: 10,
              backgroundColor: hasEnoughPoints ? "#e8f5e9" : "#ffebee",
              borderRadius: 5,
            }}
          >
            <Text
              style={{
                color: hasEnoughPoints ? "#2e7d32" : "#c62828",
                textAlign: "center",
                fontWeight: "500",
              }}
            >
              {hasEnoughPoints
                ? "Siparişi tamamlamak için yeterli puanınız var."
                : `Puanınız yetersiz. ${
                    totalPoints - userPoints
                  } adım puan daha kazanmanız gerekiyor.`}
            </Text>
          </View>
        )}
      </View>

      {/* Checkout Button */}
      <TouchableOpacity
        style={[
          styles.paymentButton,
          (isCheckingOut ||
            isSyncingCart ||
            (!selectedAddress && !newAddressMode) ||
            (newAddressMode &&
              (!formData.fullName ||
                !formData.phone ||
                !formData.fullAddress)) ||
            cartItems.length === 0 ||
            !hasEnoughPoints) && { opacity: 0.7 },
        ]}
        onPress={handleSubmit}
        disabled={
          isCheckingOut ||
          isSyncingCart ||
          (!selectedAddress && !newAddressMode) ||
          (newAddressMode &&
            (!formData.fullName || !formData.phone || !formData.fullAddress)) ||
          cartItems.length === 0 ||
          !hasEnoughPoints
        }
      >
        {isCheckingOut || isSyncingCart ? (
          <ActivityIndicator size="small" color={Colors.white} />
        ) : (
          <>
            <Feather name="check-circle" size={20} color={Colors.white} />
            <Text style={styles.paymentButtonText}>Ödemeyi Tamamla</Text>
          </>
        )}
      </TouchableOpacity>
    </View>
  );
};

export default OrderSummary;
