import React from "react";
import { View, Text, ScrollView, TouchableOpacity, Alert } from "react-native";
import { Feather, Ionicons } from "@expo/vector-icons";
import Topbar from "@/app/components/Topbar";
import Toolbar from "@/app/components/Toolbar";
import { styles, Colors } from "@/assets/styles/App.styles";
import { useNotificationsQuery } from "@/app/hooks/queries/useNotifications";
import { useMarkNotificationRead } from "@/app/hooks/mutations/useMarkNotificationRead";
import { useDeleteNotification } from "@/app/hooks/mutations/useDeleteNotification";
import moment from "moment";
import { Swipeable } from "react-native-gesture-handler";

const NotificationsScreen = () => {
  // Fetch notifications from your store
  const { data: notifications = [] } = useNotificationsQuery();

  // Define the mutation hooks at the screen level
  const markNotificationRead = useMarkNotificationRead();
  const deleteNotification = useDeleteNotification();

  // 스와이프 액션 렌더링 함수
  const renderRightActions = (notification) => {
    return (
      <TouchableOpacity
        style={{
          backgroundColor: Colors.danger,
          justifyContent: "center",
          alignItems: "center",
          width: 80,
          height: "90%",
          borderRadius: 10,
        }}
        onPress={() => {
          Alert.alert(
            "Bildirimi Sil",
            "Bu bildirimi silmek istediğinizden emin misiniz?",
            [
              {
                text: "İptal",
                style: "cancel",
              },
              {
                text: "Sil",
                onPress: () => deleteNotification.mutate(notification.id),
                style: "destructive",
              },
            ]
          );
        }}
      >
        <Ionicons name="trash-outline" size={24} color={Colors.white} />
      </TouchableOpacity>
    );
  };

  const NotificationItem = ({ notification }) => {
    // Determine which icon to show based on notification type
    const getIcon = (type) => {
      switch (type) {
        case "appointment":
          return "calendar";
        case "diet":
          return "clipboard";
        case "reminder":
          return "bell";
        default:
          return "info";
      }
    };

    return (
      <Swipeable
        renderRightActions={() => renderRightActions(notification)}
        friction={2}
        rightThreshold={40}
      >
        <TouchableOpacity
          activeOpacity={0.6}
          onPress={() => {
            // If the notification is not marked as read, call the mutation.
            if (!notification.read_at) {
              markNotificationRead.mutate(notification.id);
            }
          }}
          style={[
            styles.userHomepageAppointmentItem,
            // Show the green left border when notification.read_at is falsy (i.e. unread)
            !notification.read_at && {
              borderLeftWidth: 4,
              borderLeftColor: Colors.primary,
            },
          ]}
        >
          <View
            style={{
              flexDirection: "row",
              alignItems: "center",
              justifyContent: "space-between",
            }}
          >
            <View
              style={{ flexDirection: "row", alignItems: "center", flex: 1 }}
            >
              <View
                style={{
                  backgroundColor: Colors.primary,
                  padding: 10,
                  borderRadius: 10,
                  marginRight: 10,
                }}
              >
                <Feather
                  name={getIcon(notification.type)}
                  size={20}
                  color={Colors.white}
                />
              </View>
              <View style={{ flex: 1 }}>
                <Text
                  style={[
                    styles.userHomepageAppointmentName,
                    !notification.read_at && { fontFamily: "montbold" },
                  ]}
                >
                  {notification.title}
                </Text>
                <Text style={styles.userHomepageAppointmentDate}>
                  {notification.message}
                </Text>
                <Text
                  style={{
                    color: Colors.tertiary,
                    fontSize: 12,
                    fontFamily: "montregular",
                    marginTop: 5,
                  }}
                >
                  {(() => {
                    // Parse and display the relative time from created_at.
                    const parsedDate = moment(
                      notification.created_at,
                      "YYYY-MM-DDTHH:mm:ss",
                      true
                    );
                    if (!parsedDate.isValid()) {
                      console.error("Invalid date:", notification.created_at);
                      return "Tarih Hatası";
                    }
                    const daysPassed = moment().diff(parsedDate, "days");
                    return daysPassed <= 0 ? "Bugün" : `${daysPassed} gün önce`;
                  })()}
                </Text>
              </View>
            </View>
          </View>
        </TouchableOpacity>
      </Swipeable>
    );
  };

  return (
    <View style={styles.userHomepageContainer}>
      <Topbar title="Bildirimler" />
      <ScrollView
        style={styles.userHomepageScrollContainer}
        showsVerticalScrollIndicator={false}
      >
        <View style={styles.userHomepageGradient}>
          {notifications.length === 0 ? (
            <View style={{ padding: 20, alignItems: "center" }}>
              <Text
                style={{ fontFamily: "montmedium", color: Colors.tertiary }}
              >
                Bildirim bulunmamaktadır
              </Text>
            </View>
          ) : (
            notifications.map((notification) => (
              <NotificationItem
                key={notification.id}
                notification={notification}
              />
            ))
          )}
        </View>
      </ScrollView>
      <Toolbar />
    </View>
  );
};

export default NotificationsScreen;
