// app/hooks/mutations/useSendMessage.js
import { useMutation, useQueryClient } from "@tanstack/react-query";

// WebSocket 연결이 열릴 때까지 대기하는 헬퍼 함수 (기본 타임아웃 3초)
const waitForConnection = (ws, timeout = 5000) => {
  return new Promise((resolve, reject) => {
    // 이미 열려 있으면 즉시 resolve
    if (ws.readyState === WebSocket.OPEN) {
      resolve();
      return;
    }

    const interval = setInterval(() => {
      if (ws.readyState === WebSocket.OPEN) {
        clearInterval(interval);
        resolve();
      }
    }, 50);

    setTimeout(() => {
      clearInterval(interval);
      reject(new Error("WebSocket connection timed out"));
    }, timeout);
  });
};

const sendMessageOverWebSocket = async ({ ws, messageText }) => {
  if (!ws) {
    throw new Error("WebSocket is not initialized.");
  }

  // 만약 연결이 아직 열리지 않았다면 기다립니다.
  if (ws.readyState !== WebSocket.OPEN) {
    await waitForConnection(ws);
  }

  // JSON 형식 대신 순수 텍스트를 전송
  ws.send(messageText);

  return { success: true, message: messageText };
};

export const useSendMessageMutation = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: sendMessageOverWebSocket,
    onSuccess: () => {
      // 메시지 전송 후 채팅 목록을 새로고침하기 위해 관련 쿼리 무효화
      queryClient.invalidateQueries(["conversation"]);
      queryClient.invalidateQueries(["messageThreads"]);
    },
  });
};
