// app/hooks/mutations/useSyncSteps.js
import { useMutation, useQueryClient } from "@tanstack/react-query";
import * as stepsEndpoints from "@/app/api/endpoints/steps";
import dayjs from "dayjs";
import AsyncStorage from "@react-native-async-storage/async-storage";
import useHealthStore from "@/app/store/healthStore";
import NetInfo from "@react-native-community/netinfo";
import useAuthStore from "@/app/store/authStore";
import { addToOfflineQueue } from "@/app/utils/offlineQueueUtils";

const MAX_RETRY_ATTEMPTS = 2; // 3회에서 2회로 줄임
const RETRY_DELAY = 5000; // 5초로 유지
const MIN_SYNC_INTERVAL = 30 * 1000; // 최소 동기화 간격: 30초
const COOLDOWN_PERIOD = 5 * 60 * 1000; // 5분 쿨다운 (오류 발생 후)

// 마지막 오류 시간 체크
const isInErrorCooldown = async () => {
  try {
    const lastErrorTimeStr = await AsyncStorage.getItem(getLastErrorTimeKey());
    if (!lastErrorTimeStr) return false;

    const lastErrorTime = parseInt(lastErrorTimeStr);
    const now = Date.now();
    const timeSinceError = now - lastErrorTime;

    // 쿨다운 기간 내인지 확인
    if (timeSinceError < COOLDOWN_PERIOD) {
      console.log(
        `Error cooldown active: ${Math.round(
          timeSinceError / 1000
        )}s elapsed, cooldown is ${COOLDOWN_PERIOD / 1000}s`
      );
      return true;
    }

    return false;
  } catch (error) {
    console.error("Error checking cooldown time:", error);
    return false;
  }
};

// 오류 발생 시간 기록
const recordErrorTime = async () => {
  try {
    await AsyncStorage.setItem(getLastErrorTimeKey(), Date.now().toString());
  } catch (error) {
    console.error("Error recording error time:", error);
  }
};

// 네트워크 연결 체크 함수
const checkNetwork = async () => {
  try {
    const state = await NetInfo.fetch();
    // isInternetReachable가 null이 아닌 경우에만 값을 확인 (null은 알 수 없음)
    return state.isConnected && state.isInternetReachable !== false;
  } catch (error) {
    console.error("Error checking network:", error);
    return false;
  }
};

// 마지막 동기화 시도 시간을 체크
const canSyncNow = async () => {
  try {
    // 쿨다운 체크 (오류 발생 후 일정 시간 동안 API 호출 제한)
    const inCooldown = await isInErrorCooldown();
    if (inCooldown) {
      console.log("API 호출 쿨다운 중. 로컬에만 저장합니다.");
      return false;
    }

    const lastSyncStr = await AsyncStorage.getItem(getLastSyncAttemptKey());
    if (!lastSyncStr) return true;

    const lastSync = parseInt(lastSyncStr);
    const now = Date.now();
    const timeSinceLastSync = now - lastSync;

    // 너무 짧은 간격으로 싱크 방지
    if (timeSinceLastSync < MIN_SYNC_INTERVAL) {
      console.log(
        `Sync throttled: Last sync was ${Math.round(
          timeSinceLastSync / 1000
        )}s ago, minimum interval is ${MIN_SYNC_INTERVAL / 1000}s`
      );
      return false;
    }

    return true;
  } catch (error) {
    console.error("Error checking last sync time:", error);
    return true;
  }
};

// 마지막 동기화 시도 시간 업데이트
const updateLastSyncTime = async () => {
  try {
    await AsyncStorage.setItem(getLastSyncAttemptKey(), Date.now().toString());
  } catch (error) {
    console.error("Error updating last sync time:", error);
  }
};

// Helper function to implement retry logic with delay
const retryOperation = async (operation, maxRetries) => {
  let lastError = null;
  for (let attempt = 0; attempt < maxRetries; attempt++) {
    try {
      // 각 시도마다 네트워크 상태 확인
      const isConnected = await checkNetwork();
      if (!isConnected) {
        console.warn("네트워크 연결이 끊어졌습니다. 재시도를 건너뜁니다.");
        throw new Error("Network unavailable");
      }

      // 타임아웃 값 점진적 증가 (첫 시도는 기본 타임아웃, 이후 시도는 1.5배씩 증가)
      const timeoutMultiplier = attempt === 0 ? 1 : 1.5 * (attempt + 1);
      const result = await operation(timeoutMultiplier);
      return result;
    } catch (error) {
      lastError = error;

      // 404 에러는 재시도하지 않고 바로 오류 발생 처리 (새 레코드가 필요하기 때문)
      if (error.response?.status === 404) {
        throw error;
      }

      // 서버 응답이 있는 오류는 재시도하지 않음
      if (error.response) {
        throw error;
      }

      // 마지막 시도인 경우 오류 발생 시간 기록
      if (attempt === maxRetries - 1) {
        await recordErrorTime();
      }

      // 지수 백오프 - 각 재시도마다 대기 시간 증가
      const delay = RETRY_DELAY * Math.pow(2, attempt);
      await new Promise((resolve) => setTimeout(resolve, delay));
    }
  }
  throw lastError; // Throw the last error after all retries fail
};

// Queue to store pending steps when offline
const getOfflineQueue = async () => {
  try {
    const queueData = await AsyncStorage.getItem(getOfflineQueueKey());
    return queueData ? JSON.parse(queueData) : [];
  } catch (error) {
    console.error("Error reading offline queue:", error);
    return [];
  }
};

// 로컬 걸음수 히스토리 업데이트 함수
const updateLocalStepHistory = async (increment) => {
  try {
    // 현재 로컬 걸음수 가져오기
    const storedSteps = (await AsyncStorage.getItem("DAILY_STEP")) || "0";
    const currentSteps = parseInt(storedSteps) + increment;

    // 로컬 스토리지 업데이트
    await AsyncStorage.setItem("DAILY_STEP", String(currentSteps));
    await AsyncStorage.setItem("DAILY_DATE", dayjs().format("YYYY-MM-DD"));

    // 히스토리 데이터 가져오기
    const historyStr = await AsyncStorage.getItem(getStepHistoryKey());
    const history = historyStr ? JSON.parse(historyStr) : {};

    // 오늘 날짜
    const today = dayjs().format("YYYY-MM-DD");

    // 오늘 데이터 업데이트 또는 생성
    const todayData = history[today] || { steps: 0, date: today };
    todayData.steps = currentSteps;
    todayData.calories = Math.round(currentSteps * 0.04);
    todayData.distance = Number((currentSteps * 0.0007).toFixed(2));

    // 데일리 목표 정보 가져오기
    const dailyGoal = useHealthStore.getState().dailyGoal;
    todayData.goalAchieved = currentSteps >= dailyGoal;

    // 히스토리에 저장
    history[today] = todayData;
    await AsyncStorage.setItem(getStepHistoryKey(), JSON.stringify(history));

    console.log(
      `Local step history updated: ${increment} steps added, total: ${currentSteps}`
    );
  } catch (error) {
    console.error("Error updating local step history:", error);
  }
};

// 당일 걸음수 레코드 생성 함수 (404 오류 처리용)
const createNewStepsRecord = async (currentSteps = 0) => {
  try {
    const today = dayjs().format("YYYY-MM-DD");
    const payload = {
      record_date: today,
      steps_count: currentSteps || 0,
      points_earned: Math.floor((currentSteps || 0) / 1000) * 10,
      calories_burned: Number(((currentSteps || 0) * 0.04).toFixed(2)),
      distance_km: Number(((currentSteps || 0) * 0.0007).toFixed(2)),
      goal_achieved: false,
      sync_source: "mobile-app",
    };

    console.log("Creating new steps record with initial steps:", currentSteps);

    // 레코드 생성 시도
    const response = await stepsEndpoints.sendStepsToDB(currentSteps);
    console.log("New step record created successfully:", response);

    // 성공 시 쿨다운 초기화
    await AsyncStorage.removeItem(getLastErrorTimeKey());

    return response;
  } catch (error) {
    console.error("Error creating new step record:", error);
    // 오류 시간 기록
    await recordErrorTime();
    throw error;
  }
};

export const useSyncSteps = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ steps }) => {
      try {
        // 최소 동기화 간격 체크
        const canSync = await canSyncNow();
        if (!canSync) {
          // 최근에 동기화 시도한 경우 또는 쿨다운 중인 경우 로컬에만 저장
          await updateLocalStepHistory(steps);
          return { steps_count: 0, throttled: true };
        }

        // 동기화 시도 시간 업데이트
        await updateLastSyncTime();

        console.log("Syncing steps increment to server:", steps);

        // 네트워크 연결 확인
        const isConnected = await checkNetwork();
        if (!isConnected) {
          console.log("No network connection, saving steps locally");
          await updateLocalStepHistory(steps);
          await addToOfflineQueue(steps);
          return { steps_count: 0, offline: true };
        }

        try {
          // Try with retry logic for network errors, 타임아웃 배수를 전달
          const response = await retryOperation(
            (timeoutMultiplier) => stepsEndpoints.accumulateSteps(steps),
            MAX_RETRY_ATTEMPTS
          );

          // 성공 시 쿨다운 초기화
          await AsyncStorage.removeItem(getLastErrorTimeKey());

          // If successful, check if we have any queued steps to send
          try {
            const queue = await getOfflineQueue();
            if (queue.length > 0) {
              console.log("Processing offline queue, items:", queue.length);
              // Process queue in background
              setTimeout(async () => {
                try {
                  // 큐 처리 전에 네트워크 확인
                  const isOnline = await checkNetwork();
                  if (!isOnline) {
                    console.log("네트워크 연결 없음, 큐 처리 연기");
                    return;
                  }

                  for (const item of queue) {
                    await stepsEndpoints.accumulateSteps(item.steps);
                  }
                  // Clear queue after successful processing
                  await AsyncStorage.setItem(
                    getOfflineQueueKey(),
                    JSON.stringify([])
                  );
                  console.log("Offline queue processed successfully");

                  // 큐 처리 후 Query 무효화하여 최신 데이터 가져오게 함
                  queryClient.invalidateQueries({ queryKey: ["stepHistory"] });
                } catch (queueError) {
                  console.warn("Error processing offline queue:", queueError);
                }
              }, 100);
            }
          } catch (queueError) {
            console.warn("Error processing offline queue:", queueError);
          }

          // 응답이 유효한지 확인
          if (!response || typeof response.steps_count !== "number") {
            console.warn("Invalid sync response, using fallback:", response);
            return { steps_count: 0 };
          }

          // 로컬 히스토리 업데이트 (서버 동기화 성공 시)
          await updateLocalStepHistory(0); // 증가분 0으로 현재 값 유지하며 업데이트

          return response;
        } catch (error) {
          // 404 오류 처리 - 당일 레코드가 없는 경우
          if (error.response?.status === 404) {
            console.log("No step record found for today, creating new record");

            // 현재 저장된 걸음수 가져오기
            const storedSteps =
              (await AsyncStorage.getItem("DAILY_STEP")) || "0";
            const currentSteps = parseInt(storedSteps);

            try {
              // 새 레코드 생성 시도 (현재 걸음수로)
              const newRecord = await createNewStepsRecord(currentSteps);

              // 생성 성공 시 로컬 히스토리 업데이트
              await updateLocalStepHistory(0);

              return {
                steps_count: newRecord.steps_count || currentSteps,
                new_record: true,
              };
            } catch (createError) {
              // 생성 실패 시 오프라인 큐에 추가
              console.warn("Failed to create new step record, saving locally");
              await updateLocalStepHistory(steps);
              await addToOfflineQueue(steps);
              return { steps_count: 0, error: true, offline: true };
            }
          }

          // 타임아웃 등 네트워크 오류
          console.warn("Network error in accumulateSteps:", error.message);
          await recordErrorTime(); // 오류 시간 기록

          // 오프라인 큐에 추가 & 로컬 업데이트
          await updateLocalStepHistory(steps);
          await addToOfflineQueue(steps);

          return { steps_count: 0, error: true, offline: true };
        }
      } catch (error) {
        console.warn("Error occurred in mutation:", error);

        // 모든 경우에 로컬 히스토리는 업데이트
        await updateLocalStepHistory(steps);

        // 에러가 발생하면 fallback 값 반환
        return { steps_count: 0, error: true };
      }
    },
    onSuccess: (data) => {
      if (data.offline || data.throttled) {
        console.log(
          "Steps saved offline or throttled, skipping online updates"
        );
        return;
      }

      // 새 레코드 생성된 경우 queryClient 업데이트
      if (data.new_record) {
        console.log("New step record created, updating query cache");
        queryClient.invalidateQueries({ queryKey: ["stepHistory"] });
      }

      if (typeof data.steps_count !== "number") {
        console.warn("onSuccess received invalid data, skipping update", data);
        return;
      }

      queryClient.invalidateQueries({ queryKey: ["stepHistory"] });
      useHealthStore.setState({
        lastSyncedSteps: data.steps_count,
        lastSyncDate: new Date().toISOString(),
      });
    },
    onError: (error) => {
      // 에러가 UI에 노출되지 않도록 console.warn을 사용
      console.warn("Error syncing steps in mutation:", error);

      // 404 오류는 이미 mutationFn에서 처리했으므로 여기서는 무시
      if (error.response?.status === 404) {
        return;
      }

      // 네트워크 오류 표시
      if (!error.response) {
        console.warn("Network error in step sync.");
      }
    },
  });
};

// Helper function to save data to AsyncStorage
const saveToLocalStorage = async (payload) => {
  try {
    const historyString = await AsyncStorage.getItem(getStepHistoryKey());
    const history = historyString ? JSON.parse(historyString) : {};

    // Save using the date as the key
    history[payload.record_date] = {
      steps: payload.steps_count,
      points: payload.points_earned,
      calories: payload.calories_burned,
      distance: payload.distance_km,
      goalAchieved: payload.goal_achieved,
    };

    await AsyncStorage.setItem(getStepHistoryKey(), JSON.stringify(history));
  } catch (error) {
    console.error("Error saving to local storage:", error);
  }
};

const getUserId = () => {
  const user = useAuthStore.getState().user; // assuming user is stored in auth store
  return user?.id || "";
};

const getStepHistoryKey = () => {
  const userId = getUserId();
  return userId ? `STEP_HISTORY_${userId}` : "STEP_HISTORY";
};

const getLastSyncAttemptKey = () => {
  const userId = getUserId();
  return userId ? `LAST_SYNC_ATTEMPT_${userId}` : "LAST_SYNC_ATTEMPT";
};

const getLastErrorTimeKey = () => {
  const userId = getUserId();
  return userId ? `LAST_STEPS_ERROR_TIME_${userId}` : "LAST_STEPS_ERROR_TIME";
};

const getOfflineQueueKey = () => {
  const userId = getUserId();
  return userId ? `STEPS_OFFLINE_QUEUE_${userId}` : "STEPS_OFFLINE_QUEUE";
};
