import { useState, useEffect } from "react";
import {
  syncShoppingCartWithLocal,
  clearShoppingCart,
} from "@/app/api/endpoints/shopping-cart";
import Toast from "@/app/components/common/ToastMessage";
import useCartStore from "@/app/store/cartStore";
import api from "@/app/lib/axios";
import { useUserPoints } from "@/app/hooks/queries/useUserPoints";

export const useCartSync = () => {
  const [isSyncingCart, setIsSyncingCart] = useState(false);
  const [isCheckingPoints, setIsCheckingPoints] = useState(false);
  const cartItems = useCartStore((state) => state.items);
  const setCartItems = useCartStore((state) => state.setCartItems);

  // useUserPoints 훅을 사용하여 포인트 정보 가져오기
  const { data: pointsData, refetch: refetchPoints } = useUserPoints();

  // 백엔드와 장바구니 동기화 함수
  const syncCartWithBackend = async () => {
    if (!cartItems || cartItems.length === 0) {
      Toast("Hata", "Sepetiniz boş. Lütfen ürün ekleyin.", "danger");
      return false;
    }

    try {
      setIsSyncingCart(true);
      console.log("Syncing cart using improved method...");

      // 새로운 동기화 함수 사용
      const result = await syncShoppingCartWithLocal(cartItems);

      setIsSyncingCart(false);

      // 경고 메시지가 있으면 표시
      if (result.warning) {
        console.warn("Cart sync warning:", result.warning);
        // 백엔드 동기화 실패해도 로컬 장바구니 사용 가능하므로 성공으로 처리
      }

      return true;
    } catch (error) {
      console.error("Error syncing cart with backend:", error);
      Toast(
        "Bilgi",
        "Sepet yerel olarak kullanılabilir, ancak sunucuyla senkronize edilemedi.",
        "info"
      );
      setIsSyncingCart(false);
      // 백엔드 동기화 실패해도 로컬 장바구니는 계속 사용할 수 있으므로 true 반환
      return true;
    }
  };

  // 사용자 포인트 확인
  const checkUserPoints = async () => {
    setIsCheckingPoints(true);
    try {
      // 필요한 총 포인트 계산
      const requiredPoints = cartItems.reduce(
        (total, item) => total + item.points_required * item.quantity,
        0
      );

      // 최신 포인트 정보 가져오기
      await refetchPoints();
      const userPoints = pointsData?.total_points || 0;

      console.log(`User has ${userPoints} points, required: ${requiredPoints}`);

      setIsCheckingPoints(false);

      if (userPoints < requiredPoints) {
        Toast(
          "Yetersiz Puan",
          `Bu siparişi tamamlamak için yeterli puanınız yok. Gerekli: ${requiredPoints}, Mevcut: ${userPoints}`,
          "warning"
        );
        return {
          success: false,
          hasEnoughPoints: false,
          requiredPoints,
          availablePoints: userPoints,
        };
      }

      return {
        success: true,
        hasEnoughPoints: true,
        requiredPoints,
        availablePoints: userPoints,
      };
    } catch (error) {
      console.error("Error checking user points:", error);
      setIsCheckingPoints(false);
      return { success: false, hasEnoughPoints: false, error };
    }
  };

  // 체크아웃 성공 후 장바구니 비우기
  const clearCart = async () => {
    try {
      // 로컬 장바구니 비우기
      setCartItems([]);

      // 백엔드 장바구니 비우기 시도
      try {
        await clearShoppingCart();
        console.log("Cleared shopping cart after successful order");
      } catch (backendError) {
        console.warn(
          "Failed to clear backend cart, but local cart is cleared:",
          backendError
        );
      }

      return true;
    } catch (error) {
      console.error("Failed to clear cart:", error);
      // 로컬 장바구니는 이미 비워졌으므로 성공으로 간주
      return true;
    }
  };

  return {
    isSyncingCart,
    isCheckingPoints,
    syncCartWithBackend,
    checkUserPoints,
    clearCart,
  };
};

export default useCartSync;
