// app/hooks/useNotificationSetup.js
import { useEffect, useRef } from "react";
import * as Notifications from "expo-notifications";
import * as Device from "expo-device";
import { Platform } from "react-native";
import { useRouter } from "expo-router";
import useAuthStore from "@/app/store/authStore";
import useNotificationStore from "@/app/store/notificationStore";
// import { showMessage } from "react-native-flash-message";
import api from "@/app/lib/axios";

export const useNotificationSetup = () => {
  const router = useRouter();
  const notificationListener = useRef();
  const responseListener = useRef();
  const { fetchNotifications } = useNotificationStore();

  const registerForPushNotificationsAsync = async () => {
    if (!Device.isDevice) {
      console.log("실제 기기에서만 푸시 알림을 사용할 수 있습니다!");
      return;
    }
    const { status: existingStatus } =
      await Notifications.getPermissionsAsync();
    let finalStatus = existingStatus;
    if (existingStatus !== "granted") {
      const { status } = await Notifications.requestPermissionsAsync();
      finalStatus = status;
    }
    if (finalStatus !== "granted") {
      console.log("알림 권한이 거부되었습니다!");
      return;
    }

    // Expo 푸시 토큰 가져오기 및 서버에 등록
    try {
      const tokenData = await Notifications.getExpoPushTokenAsync({
        projectId: "28162de2-ef64-46cd-bb61-d3fde2b5f69e",
      });
      const token = tokenData.data;
      console.log("Push 토큰 (useNotificationSetup):", token);

      // 백엔드에 토큰 전송
      const authState = useAuthStore.getState();
      if (authState.isAuthenticated) {
        try {
          await api.post("/device-tokens/register", {
            device_token: token,
            device_type: Platform.OS,
          });
          console.log("디바이스 토큰이 서버에 등록되었습니다.");
        } catch (error) {
          console.error("디바이스 토큰 등록 실패:", error);
        }
      } else {
        console.log("사용자가 인증되지 않았습니다. 토큰 등록을 건너뜁니다.");
      }

      return token;
    } catch (error) {
      console.error("Expo 토큰 가져오기 실패:", error);
    }
  };

  // 백그라운드 알림 응답 처리 (사용자가 알림 탭함)
  const handleNotificationResponse = (response) => {
    console.log("알림 응답 처리 (useNotificationSetup):", response);
    try {
      const data = response.notification.request.content.data;
      console.log("알림 응답 데이터:", data);

      // 알림 데이터에 따라 적절한 화면으로 이동
      if (data?.type === "chat" && data?.chatId) {
        router.push(`/chat/${data.chatId}`);
      } else if (data?.notification_type === "test") {
        router.push("/pages/notifications");
      } else {
        // 기본적으로 알림 화면으로 이동
        router.push("/pages/notifications");
      }

      // 알림 목록 갱신
      fetchNotifications();
    } catch (error) {
      console.error("알림 응답 처리 오류:", error);
      // 오류 발생 시 기본적으로 알림 화면으로 이동
      router.push("/pages/notifications");
    }
  };

  useEffect(() => {
    console.log("useNotificationSetup 초기화");
    // 알림 권한 요청 및 토큰 등록
    registerForPushNotificationsAsync();

    // 포그라운드 알림 수신은 _layout.tsx에서 처리

    // 알림 응답 리스너 설정 (사용자가 알림 탭했을 때)
    responseListener.current =
      Notifications.addNotificationResponseReceivedListener(
        handleNotificationResponse
      );

    return () => {
      if (notificationListener.current) {
        Notifications.removeNotificationSubscription(
          notificationListener.current
        );
      }
      if (responseListener.current) {
        Notifications.removeNotificationSubscription(responseListener.current);
      }
    };
  }, []);

  // 필요한 기능 반환
  return {
    registerForPushNotificationsAsync,
    handleNotificationResponse,
  };
};

// default export 추가
export default useNotificationSetup;
