import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import api from "@/app/lib/axios";
import { getUserProfile } from "@/app/api/endpoints/users";
import useAuthStore from "@/app/store/authStore";
import Toast from "@/app/components/common/ToastMessage";
import { useNavigation } from "expo-router";

const useProfile = () => {
  const { user: authUser } = useAuthStore();
  return useQuery({
    queryKey: ["userProfile", authUser?.id],
    queryFn: () => {
      if (!authUser?.id) {
        throw new Error("User ID not found");
      }
      return getUserProfile(authUser.id);
    },
    enabled: Boolean(authUser?.id),
    // 데이터를 항상 최신 상태로 유지
    staleTime: 0, // 항상 데이터를 stale로 간주
    cacheTime: 5 * 60 * 1000, // 5분 캐시 유지 (짧게 조정)
    refetchOnMount: "always", // 항상 마운트될 때마다 새로운 데이터 가져오기
    refetchOnWindowFocus: true, // 창이 포커스를 받을 때 데이터 업데이트
    // 에러 발생 시 자동 재시도를 최소화
    retry: 1,
  });
};

const useUpdateProfile = () => {
  const queryClient = useQueryClient();
  const { user: authUser } = useAuthStore();
  const navigation = useNavigation();
  // 아래 updateProfileMutation으로 반환되는 객체의 reset() 메서드를 사용하여 에러 상태를 초기화합니다.
  const updateProfileMutation = useMutation({
    mutationFn: async (payload) => {
      return await api.put(`/users/${authUser.id}`, payload, {
        headers: { "Content-Type": "multipart/form-data" },
      });
    },
    onSuccess: () => {
      // 프로필 정보를 갱신하는 쿼리를 무효화하여 최신 데이터를 불러옵니다.
      queryClient.invalidateQueries(["userProfile", authUser?.id]);
    },
    onError: (error) => {
      // Try to check for error message in both 'message' and 'detail'
      const errorMessage = (
        error?.response?.data?.message ||
        error?.response?.data?.detail ||
        ""
      ).toString();
      if (
        errorMessage.includes("Input should be greater than or equal to 5000")
      ) {
        // Show Turkish Toast message and navigate back
        Toast("Hata", "Giriş 5000 veya daha büyük olmalıdır.", "danger");
        setTimeout(() => {
          updateProfileMutation.reset();
          navigation.goBack();
        }, 2000);
      } else {
        // For other errors, simply reset the mutation error state without showing any toast
        updateProfileMutation.reset();
      }
    },
  });
  return updateProfileMutation;
};

export { useProfile as useUserProfile, useUpdateProfile };

// default export 추가
export default useProfile;
