import React from "react";
import useCartStore from "@/app/store/cartStore";

// A hook to retrieve the shopping cart state
export const useShoppingCart = () => {
  const items = useCartStore((state) => state.items);
  return { data: { items }, isLoading: false, error: null };
};

// A hook to add an item to the shopping cart (temporary storage - verified by backend during payment)
export const useAddCartItem = () => {
  const addItem = useCartStore((state) => state.addItem);
  return {
    mutate: (product) => {
      try {
        if (!product || !product.id) {
          console.error("Invalid product data:", product);
          return;
        }
        addItem(product);
      } catch (error) {
        console.error("Error adding item to cart:", error);
      }
    },
    isLoading: false,
    error: null,
  };
};

// A hook to remove an item from the shopping cart
export const useRemoveCartItem = () => {
  const removeItem = useCartStore((state) => state.removeItem);
  return {
    mutate: (productId) => {
      try {
        if (!productId) {
          console.error("Invalid product ID for removal");
          return;
        }
        removeItem(productId);
      } catch (error) {
        console.error("Error removing item from cart:", error);
      }
    },
    isLoading: false,
    error: null,
  };
};

// A hook to update the quantity of an item in the shopping cart
export const useUpdateCartItem = () => {
  const updateItemQuantity = useCartStore((state) => state.updateItemQuantity);
  const addItem = useCartStore((state) => state.addItem);

  return {
    mutate: ({ productId, quantity, product }) => {
      try {
        if (!productId) {
          console.error("Invalid product ID for update");
          return;
        }

        if (product) {
          // product 객체가 제공되면 명시적 업데이트 모드로 addItem 사용
          console.log(
            `Updating cart item with full product object: ${productId} quantity ${quantity}`
          );
          addItem({
            ...product,
            id: productId,
            quantity: quantity,
            updateMode: true, // 명시적 업데이트 모드 플래그
          });
        } else {
          // 기존 방식 (단순 quantity 업데이트)
          console.log(
            `Updating cart item quantity: ${productId} to ${quantity}`
          );
          updateItemQuantity(productId, quantity);
        }
      } catch (error) {
        console.error("Error updating cart item:", error);
      }
    },
    isLoading: false,
    error: null,
  };
};

// A hook to synchronize the shopping cart with the updated stock information from the backend (optional)
export const useSyncCartStocks = (updatedStocks) => {
  const syncCartStocks = useCartStore((state) => state.syncCartStocks);

  React.useEffect(() => {
    if (updatedStocks) {
      try {
        syncCartStocks(updatedStocks);
      } catch (error) {
        console.error("Error syncing cart stocks:", error);
      }
    }
  }, [updatedStocks, syncCartStocks]);

  return { isLoading: false, error: null };
};
