import { useQuery } from "@tanstack/react-query";
import api from "@/app/lib/axios";
import useAuthStore from "@/app/store/authStore";
import { Platform } from "react-native";

/**
 * 사용자 포인트 정보를 가져오는 함수
 * 백엔드에서 받는 응답 형식에 따라 일관된 구조로 변환합니다.
 */
const fetchUserPoints = async (token) => {
  if (!token) throw new Error("Missing token");

  try {
    // 포인트 전용 엔드포인트로 요청
    const response = await api.get(`/points/`, {
      headers: { Authorization: `Bearer ${token}` },
    });

    // 백엔드 응답이 다양한 형태로 올 수 있어 처리
    if (response.data && typeof response.data === "object") {
      // 이미 total_points 필드가 있는 경우
      if ("total_points" in response.data) {
        return response.data;
      }

      // points 필드가 있는 경우
      if ("points" in response.data) {
        return {
          total_points: response.data.points,
          ...response.data,
        };
      }

      // 숫자만 반환되는 경우
      if (typeof response.data === "number") {
        return { total_points: response.data };
      }
    }

    // 기본값: 응답이 예상과 다를 경우
    return { total_points: 0 };
  } catch (error) {
    console.error("Error in fetchUserPoints:", error);

    // 폴백: 사용자 정보 API로 대체 시도
    try {
      const userResponse = await api.get("/users/me/", {
        headers: { Authorization: `Bearer ${token}` },
      });

      // 개발 환경에서만 로그를 출력합니다
      if (__DEV__) {
        console.log("Fallback to user API response:", userResponse.data);
      }

      return {
        total_points: userResponse.data.points || 0,
        fallback: true,
      };
    } catch (fallbackError) {
      console.error("Fallback also failed:", fallbackError);
      throw error; // 원래 오류를 던짐
    }
  }
};

export const useUserPoints = () => {
  const { token } = useAuthStore();

  return useQuery({
    queryKey: ["userPoints", token],
    queryFn: () => fetchUserPoints(token),
    enabled: !!token,
    refetchOnMount: true, // 'always'에서 true로 변경하여 불필요한 리패치 감소
    staleTime: 30000, // 10초에서 30초로 캐시 유지 시간 증가
    onError: (error) => console.error("Error fetching user points:", error),
    select: (data) => {
      // 데이터가 일관된 형식으로 반환되도록 보장
      if (!data) return { total_points: 0 };

      return data;
    },
  });
};

// default export 추가
export default useUserPoints;
