// app/lib/axios.js
import axios from "axios";
import useAuthStore from "@/app/store/authStore";
import NetInfo from "@react-native-community/netinfo";
import AsyncStorage from "@react-native-async-storage/async-storage";
import { Platform } from "react-native";

// 서버 주소를 확인하세요! 로컬 네트워크에서 올바른 IP와 포트인지 확인
const SERVER_ADDRESS = "https://adimsayar.easerver.net/api/v1";

// 짧은 타임아웃으로 시작하고 재시도마다 증가 - 전체적으로 타임아웃 단축
const INITIAL_TIMEOUT = Platform.OS === "android" ? 15000 : 10000; // 안드로이드는 더 긴 타임아웃
const MAX_TIMEOUT = 30000; // 최대 30초

// 장바구니 및 체크아웃 관련 요청에 더 짧은 타임아웃 적용
const CART_ENDPOINTS = [
  "/shopping-cart/",
  "/shopping-cart/add",
  "/shopping-cart/remove",
  "/orders/checkout",
];
const CART_TIMEOUT = 8000; // 장바구니 요청 8초 타임아웃

// 마지막 네트워크 상태를 저장
let lastNetworkStatus = true;

const api = axios.create({
  baseURL: SERVER_ADDRESS,
  headers: {
    "Content-Type": "application/json",
    Accept: "application/json",
    Connection: "keep-alive",
    "Cache-Control": "no-cache",
    "User-Agent": `AdimsayarApp/${Platform.OS}`,
  },
  timeout: INITIAL_TIMEOUT,
  // 연결 재시도 옵션
  retries: 3,
  retryDelay: 1000,
  // APK 환경에서 중요: HTTP 응답 압축 해제
  decompress: true,
});

// 네트워크 상태 확인 함수
const checkNetwork = async () => {
  try {
    const state = await NetInfo.fetch();
    const isConnected =
      state.isConnected && state.isInternetReachable !== false;

    // 네트워크 상태가 변경되었을 때만 로그 출력
    if (isConnected !== lastNetworkStatus) {
      console.log(
        `Network status changed: ${isConnected ? "Connected" : "Disconnected"}`
      );
      lastNetworkStatus = isConnected;
    }

    return isConnected;
  } catch (error) {
    console.error("네트워크 상태 확인 오류:", error);
    return true; // 오류 시 기본적으로 연결된 것으로 가정
  }
};

// 재시도 로직을 추가한 인터셉터
api.interceptors.response.use(null, async (error) => {
  const { config, message } = error;

  // 설정 객체가 없으면 재시도 불가능
  if (!config) {
    console.warn(`[API 오류] 설정 객체가 없음: ${message}`);
    return Promise.reject(error);
  }

  // 이미 재시도 중인 요청인지 확인
  if (config._retryCount === undefined) {
    config._retryCount = 0;
  } else {
    config._retryCount++;
  }

  // 네트워크 상태 확인
  const isNetworkAvailable = await checkNetwork();
  if (!isNetworkAvailable) {
    console.warn(`[네트워크 연결 없음] ${config.url}`);
    error.isNetworkError = true;
    return Promise.reject(error);
  }

  // 점진적으로 타임아웃 증가
  const nextTimeout = Math.min(
    INITIAL_TIMEOUT * (config._retryCount + 1),
    MAX_TIMEOUT
  );

  // 재시도 로직 (네트워크 오류나 타임아웃에만 적용)
  if (
    config._retryCount < api.defaults.retries &&
    (error.code === "ECONNABORTED" || !error.response)
  ) {
    console.log(
      `요청 재시도 중... (${config._retryCount + 1}/${api.defaults.retries}): ${
        config.url
      }, 타임아웃: ${nextTimeout}ms`
    );

    // 타임아웃 증가
    config.timeout = nextTimeout;

    // 재시도 전 지연
    const retryDelay = api.defaults.retryDelay * (config._retryCount + 1);
    await new Promise((resolve) => setTimeout(resolve, retryDelay));
    return api(config);
  }

  // 자세한 오류 정보 기록
  if (error.code === "ECONNABORTED") {
    console.warn(`[Network Timeout] ${config.url || "unknown"}: ${message}`);
  } else if (!error.response) {
    console.warn(`[Network Error] ${config.url || "unknown"}: ${message}`);
  } else {
    console.warn(
      `[Server Error] ${config.url || "unknown"}: ${
        error.response?.status
      } ${message}`
    );
  }

  // 인증 오류 처리
  if (error.response?.status === 401) {
    await useAuthStore.getState().logout();
  }

  return Promise.reject(error);
});

// 요청 인터셉터
api.interceptors.request.use(
  async (config) => {
    // 네트워크 연결 확인
    const isNetworkAvailable = await checkNetwork();
    if (!isNetworkAvailable) {
      console.warn(`[No Network Connection] Request cancelled: ${config.url}`);
      // axios 요청이 취소되도록 설정
      return Promise.reject({
        isNetworkError: true,
        message: "No network connection available.",
      });
    }

    // 장바구니 및 체크아웃 관련 엔드포인트에 더 짧은 타임아웃 적용
    if (config.url) {
      const isCartEndpoint = CART_ENDPOINTS.some((endpoint) =>
        config.url.includes(endpoint)
      );

      if (isCartEndpoint) {
        console.log(
          `Applying shorter timeout (${CART_TIMEOUT}ms) for cart request: ${config.url}`
        );
        config.timeout = CART_TIMEOUT;
      }
    }

    // 인증 토큰 추가
    const token = useAuthStore.getState().token;
    if (token) {
      config.headers.Authorization = `Bearer ${token}`;
    }
    config.metadata = { startTime: new Date() };

    // 태블릿/큰 화면 디바이스의 경우 더 나은 이미지 품질 요청
    const deviceType = Platform.isPad ? "tablet" : "phone";
    config.headers["X-Device-Type"] = deviceType;

    // 안드로이드 APK 빌드인 경우 특별 헤더 추가
    if (Platform.OS === "android" && !__DEV__) {
      config.headers["X-App-Type"] = "android-production";
    }

    return config;
  },
  (error) => Promise.reject(error)
);

export default api;
