// app/pages/friends/chatPage.js
import React, {
  useState,
  useCallback,
  useEffect,
  useMemo,
  useRef,
} from "react";
import { Ionicons } from "@expo/vector-icons";
import { useRoute, useNavigation } from "@react-navigation/native";
import * as ImagePicker from "expo-image-picker";
import { Colors, styles } from "@/assets/styles/App.styles";
import {
  Modal,
  TouchableOpacity,
  View,
  Image,
  Text,
  KeyboardAvoidingView,
  Platform,
  LayoutAnimation,
  ActivityIndicator,
  Keyboard,
  SafeAreaView,
  Dimensions,
  FlatList,
} from "react-native";
import tr from "dayjs/locale/tr";
import { ChatHeader } from "@/app/components/ChatHeader";
import { ChatInput } from "@/app/components/ChatInput";
import Toast from "@/app/components/common/ToastMessage";
import { useConversation } from "@/app/hooks/queries/useConversation";
import { useSendMessageMutation } from "@/app/hooks/mutations/useSendMessage";
import useAuthStore from "@/app/store/authStore";
import { useUserProfile } from "@/app/hooks/queries/useProfile";
import Constants from "expo-constants";
import { MessageBubble } from "@/app/components/MessageBubble";
import { useMarkMessagesReadMutation } from "@/app/hooks/mutations/useMarkMessagesRead";
import { useQueryClient } from "@tanstack/react-query";

// WebSocket URL - wss:// 프로토콜 사용 (보안 연결)
const WS_HOST =
  Constants.manifest?.extra?.DEV_WS_URL || "wss://adimsayar.easerver.net";

// Helper function: 전달받은 문자열이 JSON 형식이면 파싱하여 text 프로퍼티를 반환, 아니면 원본을 반환
const parseWSMessage = (dataString) => {
  // 간단한 체크: 문자열이 "{"로 시작하고 "}"로 끝나면 JSON 형식일 가능성이 있음
  if (!dataString.trim().startsWith("{") || !dataString.trim().endsWith("}")) {
    return dataString;
  }
  try {
    const result = JSON.parse(dataString);
    // 만약 result가 객체이고 text 프로퍼티가 있다면 plain text를 반환
    return result && result.text ? result.text : dataString;
  } catch (e) {
    // 에러 로그를 제거하거나 개발환경에서만 로깅
    // if (process.env.NODE_ENV === "development") {
    //   console.error("Parsing error in parseWSMessage:", e);
    // }
    return dataString;
  }
};

// 화면 높이의 20%로 입력창 높이와 동일하게 bottom offset을 사용합니다.
const { height: screenHeight } = Dimensions.get("window");
const inputToolbarHeight = screenHeight * 0.2;

const ChatPage = () => {
  const [messages, setMessages] = useState([]);
  const [pickedImage, setPickedImage] = useState(null);
  const [isLoading, setIsLoading] = React.useState(false);
  const route = useRoute();
  const { friend } = route?.params;
  const queryClient = useQueryClient();

  const chatData = useMemo(() => JSON.parse(friend), [friend]);

  const textInputRef = useRef(null);

  const navigation = useNavigation();

  // authStore에서 사용자 정보 추출
  const { token } = useAuthStore();
  const { data: user } = useUserProfile();
  const currentUserId = user?.id; // 주의: currentUserId 의 타입이 숫자/문자열인지 확인해주세요.

  const wsRef = useRef(null);

  // React Query를 사용해 대화 내역을 로드합니다.
  const {
    data: conversation,
    refetch: refetchConversation,
    isLoading: conversationLoading,
  } = useConversation(chatData.id);

  // 읽음 처리를 위한 Mutation 훅
  const markMessagesReadMutation = useMarkMessagesReadMutation();

  // 연결 상태를 관리하는 상태 변수
  const [wsConnected, setWsConnected] = useState(false);
  const [websocketError, setWebsocketError] = useState(false);

  const flatListRef = useRef(null);

  // 컴포넌트 마운트 시 대화 데이터 강제 새로고침
  useEffect(() => {
    if (token && chatData?.id) {
      queryClient.invalidateQueries(["conversation", chatData.id]);
      refetchConversation();
    }
  }, [queryClient, token, chatData?.id]);

  useEffect(() => {
    if (conversation) {
      const formattedMessages = conversation
        .map((msg) => ({
          _id: msg.id,
          text: msg.content,
          createdAt: new Date(msg.sent_at),
          user: {
            _id: msg.sender_id,
            name:
              msg.sender_id.toString() === currentUserId.toString()
                ? "Siz"
                : chatData.name,
            avatar:
              msg.sender_id.toString() === currentUserId.toString()
                ? user.profile_pic
                : chatData.avatar,
          },
        }))
        .sort((a, b) => b.createdAt - a.createdAt);
      setMessages(formattedMessages);
    }
  }, [conversation, currentUserId, chatData, user]);

  // 대화가 로드되면, 대화 상대(chatData.id)의 읽지 않은 메시지가 있다면 읽음 처리 호출
  useEffect(() => {
    if (
      conversation &&
      Array.isArray(conversation) &&
      conversation.length > 0
    ) {
      const unreadMessages = conversation.filter(
        (msg) =>
          msg.sender_id.toString() === chatData.id.toString() && !msg.is_read
      );
      if (unreadMessages.length > 0) {
        markMessagesReadMutation.mutate(chatData.id);
      }
    }
  }, [conversation, chatData.id, markMessagesReadMutation]);

  // WebSocket 연결 설정 - 연결 실패 시 재시도 로직 추가
  const setupWebSocket = useCallback(() => {
    if (!token || !chatData?.id) return;

    try {
      const wsUrl = `${WS_HOST}/api/v1/ws-chat/ws/chat/${
        chatData.id
      }?token=${encodeURIComponent(token)}`;
      console.log("Connecting to WS:", wsUrl);

      if (wsRef.current) {
        // Close existing connection if any
        wsRef.current.close();
      }

      const ws = new WebSocket(wsUrl);
      wsRef.current = ws;

      ws.onopen = () => {
        console.log("WebSocket connected");
        setWsConnected(true);
        setWebsocketError(false);
      };

      ws.onmessage = (event) => {
        console.log("Received message via WS:", event.data);

        let parsed = null;
        try {
          parsed = JSON.parse(event.data);
        } catch (e) {
          console.log("Parsing error:", e);
          return;
        }

        // 문제 1: 자신이 보낸 메시지는 이미 로컬 상태에 추가됐으므로 무시합니다.
        if (parsed.sender_id.toString() === currentUserId.toString()) {
          return;
        }

        // 문제 2: 상대방 메시지는 이름 없이 메시지만 표시
        const receivedMessage = {
          _id: parsed.id || Date.now().toString(),
          text: parsed.content,
          createdAt: new Date(parsed.sent_at || Date.now()),
          user: {
            _id: parsed.sender_id,
            name: "", // 상대방 메시지에서는 이름을 출력하지 않음
            avatar: chatData.avatar,
          },
        };

        // 중복 메시지가 상태에 없을 경우에만 추가합니다.
        setMessages((prevMessages) => {
          const exists = prevMessages.some(
            (msg) => msg._id === receivedMessage._id
          );
          return exists ? prevMessages : [receivedMessage, ...prevMessages];
        });

        // 새 메시지를 받으면 대화 내역 새로고침
        refetchConversation();
      };

      ws.onerror = (error) => {
        console.error("WebSocket error:", error);
        setWebsocketError(true);
        setWsConnected(false);
        Toast("WebSocket Hatası", "Bağlantıda bir hata oluştu.", "danger");
      };

      ws.onclose = (e) => {
        console.log("WebSocket closed:", e);
        setWsConnected(false);
      };
    } catch (error) {
      console.error("Error setting up WebSocket:", error);
      setWebsocketError(true);
      setWsConnected(false);
    }
  }, [token, chatData?.id, currentUserId, refetchConversation]);

  // 컴포넌트 마운트/언마운트 시 WebSocket 연결 관리
  useEffect(() => {
    setupWebSocket();

    return () => {
      if (wsRef.current) {
        wsRef.current.close();
        wsRef.current = null;
      }
    };
  }, [setupWebSocket]);

  // WebSocket 오류 시 자동 재연결 시도
  useEffect(() => {
    if (websocketError) {
      const reconnectTimeout = setTimeout(() => {
        console.log("Attempting to reconnect WebSocket...");
        setupWebSocket();
      }, 3000); // 3초 후 재연결 시도

      return () => clearTimeout(reconnectTimeout);
    }
  }, [websocketError, setupWebSocket]);

  // mutation 훅을 통해 웹소켓으로 메시지 전송
  const { mutate: sendMessageMutation } = useSendMessageMutation();

  const handleSend = (messageText) => {
    if (!messageText.trim()) return;

    // UI에 표시할 메시지 객체
    const newMessage = {
      _id: Date.now().toString(),
      text: messageText,
      createdAt: new Date(),
      user: {
        _id: currentUserId,
        name: "Siz",
        avatar: user?.profile_pic,
      },
    };

    // UI 업데이트
    setMessages((previousMessages) => [newMessage, ...previousMessages]);

    // WebSocket 연결 확인 후 메시지 전송
    if (wsConnected && wsRef.current) {
      sendMessageMutation(
        {
          ws: wsRef.current,
          messageText: messageText, // 순수 텍스트만 전송
        },
        {
          onSuccess: () => {
            // 메시지 전송 성공 시 대화 목록 새로고침
            refetchConversation();
          },
          onError: (err) => {
            Toast("Mesaj Gönderilemedi", err.message, "danger");
            // 필요한 경우 WebSocket 재연결 시도
            setupWebSocket();
          },
        }
      );
    } else {
      Toast("Bağlantı Hatası", "Mesaj gönderilemedi. Bağlantı yok.", "danger");
      // WebSocket 재연결 시도
      setupWebSocket();
    }
  };

  // 로딩 상태 또는 오류 상태 처리
  if (isLoading || conversationLoading) {
    return (
      <View
        style={{
          flex: 1,
          justifyContent: "center",
          alignItems: "center",
          backgroundColor: Colors.white,
        }}
      >
        <ActivityIndicator size="large" color={Colors.primary} />
        <Text style={{ marginTop: 10, color: Colors.tertiary }}>
          Mesaj yükleniyor...
        </Text>
      </View>
    );
  }

  // 친구 데이터 누락 오류 처리
  if (!chatData) {
    return (
      <View
        style={{
          flex: 1,
          justifyContent: "center",
          alignItems: "center",
          backgroundColor: Colors.white,
        }}
      >
        <Text style={{ color: Colors.danger, marginBottom: 20 }}>
          Arkadaş bilgisi eksik veya geçersiz
        </Text>
        <TouchableOpacity
          style={{
            backgroundColor: Colors.primary,
            padding: 12,
            borderRadius: 8,
          }}
          onPress={() => navigation.goBack()}
        >
          <Text style={{ color: Colors.white }}>Geri Dön</Text>
        </TouchableOpacity>
      </View>
    );
  }

  return (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === "ios" ? "padding" : undefined}
      keyboardVerticalOffset={Platform.OS === "ios" ? 90 : 0}
    >
      <View
        style={[
          styles.ChatPageContainer,
          Platform.OS === "android" && { paddingBottom: 0 },
        ]}
      >
        <SafeAreaView edges={["top"]}>
          <ChatHeader friend={chatData} onBack={() => navigation.goBack()} />
        </SafeAreaView>

        <View style={{ flex: 1 }}>
          <FlatList
            data={messages}
            renderItem={({ item }) => (
              <MessageBubble
                message={item}
                isOwn={item.user._id === currentUserId}
              />
            )}
            keyExtractor={(item) => item._id}
            contentContainerStyle={{
              paddingVertical: 10,
            }}
            inverted={true}
            initialScrollIndex={0}
            onLayout={() => {
              if (messages.length > 0) {
                flatListRef.current?.scrollToOffset({
                  offset: 0,
                  animated: true,
                });
              }
            }}
            ref={flatListRef}
            onRefresh={refetchConversation}
            refreshing={conversationLoading}
          />
        </View>

        {!wsConnected && (
          <View
            style={{
              padding: 8,
              backgroundColor: Colors.warning,
              alignItems: "center",
              justifyContent: "center",
            }}
          >
            <Text style={{ color: Colors.white, fontSize: 12 }}>
              Bağlantı kuruluyor... Mesajlar gecikebilir.
            </Text>
          </View>
        )}

        <View
          style={[
            styles.chatInputWrapper,
            Platform.OS === "android" && {
              position: "relative",
              bottom: 0,
              left: 0,
              right: 0,
              backgroundColor: Colors.white,
              elevation: 8,
            },
          ]}
        >
          <ChatInput
            onSend={handleSend}
            pickedImage={pickedImage}
            onPickImage={() => {
              // 이미지 선택 로직
            }}
            onClearPickedImage={() => setPickedImage(null)}
          />
        </View>
      </View>
    </KeyboardAvoidingView>
  );
};

export default ChatPage;
