import React, { useEffect } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  Image,
  ActivityIndicator,
} from "react-native";
import Topbar from "@/app/components/Topbar";
import Toolbar from "@/app/components/Toolbar";
import { useRouter } from "expo-router";
import dayjs from "dayjs";
import relativeTime from "dayjs/plugin/relativeTime";
import "dayjs/locale/tr";
import { styles, Colors } from "@/assets/styles/App.styles";
import {
  useFriendships,
  useMessageThreads,
} from "@/app/hooks/queries/useFriends";
import { useUnreadMessages } from "@/app/hooks/queries/useUnreadMessages";
import useAuthStore from "@/app/store/authStore";
import Toast from "@/app/components/common/ToastMessage";
import { Feather } from "@expo/vector-icons";
import {
  useAcceptFriendship,
  useRejectFriendship,
} from "@/app/hooks/mutations/useFriendshipActions";
import { getImageUrl } from "@/app/utils/imageUtils";
import { useQueryClient } from "@tanstack/react-query";

dayjs.extend(relativeTime);
dayjs.locale("tr");

const MessagesScreen = () => {
  const router = useRouter();
  const { token, user } = useAuthStore();
  const currentUserId = user?.id;
  const queryClient = useQueryClient();

  // 컴포넌트 마운트 시 데이터 강제 새로고침
  useEffect(() => {
    if (token && currentUserId) {
      queryClient.invalidateQueries(["friendships"]);
      queryClient.invalidateQueries(["messageThreads"]);
      queryClient.invalidateQueries(["unreadMessages"]);
    }
  }, [queryClient, token, currentUserId]);

  // If the token or user information is not yet available, display a loading indicator.
  if (!token || !currentUserId) {
    return (
      <View style={styles.userHomepageContainer}>
        <ActivityIndicator size="large" color={Colors.primary} />
      </View>
    );
  }

  const {
    data: friendships,
    isLoading: friendshipsLoading,
    error: friendshipsError,
    refetch: refetchFriendships,
  } = useFriendships();

  const {
    data: messageThreads,
    isLoading: threadsLoading,
    error: threadsError,
    refetch: refetchMessageThreads,
  } = useMessageThreads();

  const {
    data: unreadMessages,
    isLoading: unreadLoading,
    error: unreadError,
    refetch: refetchUnreadMessages,
  } = useUnreadMessages();

  // 모든 데이터를 한 번에 새로고침하는 함수
  const refreshAllData = () => {
    refetchFriendships();
    refetchMessageThreads();
    refetchUnreadMessages();
  };

  // 컴포넌트가 화면에 표시될 때 자동으로 refreshAllData 호출
  useEffect(() => {
    refreshAllData();
  }, []);

  if (friendshipsLoading || threadsLoading || unreadLoading) {
    return (
      <View style={styles.userHomepageContainer}>
        <ActivityIndicator size="large" color={Colors.primary} />
      </View>
    );
  }

  if (friendshipsError || threadsError || unreadError) {
    return (
      <View style={styles.userHomepageContainer}>
        <Text>Error loading data</Text>
        <TouchableOpacity
          style={{
            marginTop: 10,
            padding: 10,
            backgroundColor: Colors.primary,
            borderRadius: 5,
          }}
          onPress={refreshAllData}
        >
          <Text style={{ color: Colors.white }}>Tekrar Dene</Text>
        </TouchableOpacity>
      </View>
    );
  }

  const combinedData = (friendships || [])
    .filter((friend) => friend.status !== "blocked")
    .map((friend) => {
      const thread = (messageThreads || []).find(
        (msg) =>
          (msg.sender_id.toString() === friend.friend_id.toString() &&
            msg.receiver_id.toString() === currentUserId.toString()) ||
          (msg.receiver_id.toString() === friend.friend_id.toString() &&
            msg.sender_id.toString() === currentUserId.toString())
      );
      const unreadCount = (unreadMessages || []).filter(
        (msg) =>
          msg.sender_id.toString() === friend.friend_id.toString() &&
          msg.receiver_id.toString() === currentUserId.toString() &&
          !msg.is_read
      ).length;

      return {
        ...friend,
        thread: thread || null,
        unreadCount,
      };
    });

  const MessageItem = ({ friendData }) => {
    const router = useRouter();
    const { user } = useAuthStore();
    const currentUserId = user?.id;

    // API 응답에 따른 필드 (응답에 새롭게 추가된 "id"는 친구 요청 고유 식별자입니다)
    const {
      id, // 친구 요청 고유 식별자 (예: 12)
      friend_id,
      first_name,
      last_name,
      profile_image,
      status,
      is_request_received, // 요청 받은 쪽이면 true
      thread,
      unreadCount,
    } = friendData;

    const friendObject = {
      id: friend_id, // 채팅 페이지 등에서는 실제 친구의 ID (요청을 보낸 사람)
      name: `${first_name} ${last_name}`,
      avatar: getImageUrl(profile_image, "profile/default_avatar.png"),
    };

    const { mutate: acceptFriendMutation } = useAcceptFriendship();
    const { mutate: rejectFriendMutation } = useRejectFriendship();

    return (
      <TouchableOpacity
        activeOpacity={0.6}
        onPress={() => {
          // 만약 친구 요청 상태가 pending 이라면 채팅 페이지로 이동하지 않고 Toast 메시지를 출력
          if (status === "pending") {
            Toast(
              "Uyarı",
              "Arkadaşlık isteğiniz henüz kabul edilmedi",
              "danger"
            );
            return;
          }

          try {
            // 채팅 페이지로 라우팅하기 전에 필요한 정보가 모두 있는지 확인
            if (!friendObject || !friendObject.id) {
              throw new Error("Arkadaş bilgileri eksik");
            }

            // 수정된 부분: forgotPassword.js와 동일한 형식으로 변경
            router.push({
              pathname: "pages/friends/chatPage",
              params: { friend: JSON.stringify(friendObject) },
            });
          } catch (error) {
            console.error("Arkadaş sayfasına giderken bir hata oluştu:", error);
            Toast(
              "Hata",
              "Arkadaş sayfasına giderken bir hata oluştu. Lütfen tekrar deneyin.",
              "danger"
            );
          }
        }}
        style={{
          backgroundColor: Colors.white,
          padding: 15,
          borderRadius: 10,
          marginBottom: 10,
          shadowColor: Colors.tertiary,
          shadowOffset: { width: 0, height: 2 },
          shadowOpacity: 0.1,
          shadowRadius: 4,
          elevation: 3,
        }}
      >
        <View style={{ flexDirection: "row", alignItems: "center" }}>
          <View style={{ marginRight: 15 }}>
            <Image
              source={{ uri: friendObject.avatar }}
              style={{
                width: 80,
                height: 80,
                borderRadius: 20,
                borderWidth: 2,
                borderColor: Colors.primary,
              }}
              onError={(e) => {
                console.error(
                  "Failed to load profile image for friend",
                  e.nativeEvent.error
                );
              }}
              defaultSource={require("@/assets/images/default_avatar.png")}
            />
          </View>
          <View style={{ flex: 1 }}>
            <Text
              style={{
                fontSize: 16,
                color: Colors.tertiary,
                fontFamily: "montbold",
                textAlign: "right",
              }}
            >
              {friendObject.name}
            </Text>
            {/* 메시지 출력 부분 */}
            {status === "pending" ? (
              <Text
                style={{
                  fontSize: 12,
                  color: Colors.warning,
                  fontFamily: "montregular",
                  flex: 1,
                  textAlign: "right",
                }}
                numberOfLines={1}
              >
                Arkadaşlık isteğiniz henüz kabul edilmedi
              </Text>
            ) : thread ? (
              <>
                <View
                  style={{
                    flexDirection: "row",
                    justifyContent: "space-between",
                    alignItems: "center",
                  }}
                >
                  <Text
                    style={{
                      fontSize: 12,
                      color: Colors.primary,
                      fontFamily: "montregular",
                      textAlign: "right",
                    }}
                  >
                    {dayjs(thread.sent_at).fromNow()}
                  </Text>
                  {unreadCount > 0 && (
                    <View
                      style={{
                        minWidth: 20,
                        height: 20,
                        borderRadius: 10,
                        backgroundColor: Colors.danger,
                        justifyContent: "center",
                        alignItems: "center",
                        marginLeft: 5,
                      }}
                    >
                      <Text
                        style={{
                          color: Colors.white,
                          fontSize: 12,
                          fontFamily: "montbold",
                        }}
                      >
                        {unreadCount}
                      </Text>
                    </View>
                  )}
                </View>
                <Text
                  style={{
                    fontSize: 14,
                    color: Colors.tertiary,
                    fontFamily: "montregular",
                    flex: 1,
                    textAlign: "right",
                  }}
                  numberOfLines={1}
                >
                  {thread.content}
                </Text>
              </>
            ) : (
              <Text
                style={{
                  fontSize: 12,
                  color: Colors.tertiary,
                  fontFamily: "montregular",
                  textAlign: "right",
                }}
              >
                Mesaj yok.
              </Text>
            )}
            {/* 친구 요청 수락/거절 버튼 */}
            {status === "pending" && is_request_received && (
              <View
                style={{
                  flexDirection: "row",
                  justifyContent: "flex-end",
                  alignItems: "center",
                  marginTop: 10,
                }}
              >
                <TouchableOpacity
                  onPress={() => {
                    console.log("요청 수락을 위한 id:", friend_id);
                    acceptFriendMutation(friend_id, {
                      onSuccess: () => {
                        refreshAllData(); // 친구 관계 업데이트 후 데이터 새로고침
                      },
                    });
                  }}
                  style={{
                    flexDirection: "row",
                    alignItems: "center",
                    marginRight: 10,
                    paddingVertical: 5,
                    paddingHorizontal: 10,
                    backgroundColor: "#4CAF50",
                    borderRadius: 5,
                  }}
                >
                  <Feather name="check" size={14} color={Colors.white} />
                  <Text style={{ color: "#fff", fontSize: 12, marginLeft: 5 }}>
                    Kabul Et
                  </Text>
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={() => {
                    console.log("요청 거절을 위한 id:", friend_id);
                    rejectFriendMutation(friend_id, {
                      onSuccess: () => {
                        refreshAllData(); // 친구 관계 업데이트 후 데이터 새로고침
                      },
                    });
                  }}
                  style={{
                    flexDirection: "row",
                    alignItems: "center",
                    paddingVertical: 5,
                    paddingHorizontal: 10,
                    backgroundColor: Colors.danger,
                    borderRadius: 5,
                  }}
                >
                  <Feather name="x" size={14} color={Colors.white} />
                  <Text style={{ color: "#fff", fontSize: 12, marginLeft: 5 }}>
                    Reddet
                  </Text>
                </TouchableOpacity>
              </View>
            )}
          </View>
        </View>
      </TouchableOpacity>
    );
  };

  return (
    <View style={styles.userHomepageContainer}>
      <Topbar title="Arkadaşlar" />
      <ScrollView
        style={styles.userHomepageScrollContainer}
        showsVerticalScrollIndicator={false}
        onRefresh={refreshAllData}
        refreshing={friendshipsLoading || threadsLoading || unreadLoading}
      >
        <View style={styles.userHomepageGradient}>
          {combinedData.map((friendData) => (
            <MessageItem key={friendData.friend_id} friendData={friendData} />
          ))}
        </View>
      </ScrollView>
      <Toolbar selectedMenu={3} />
    </View>
  );
};

export default MessagesScreen;
