import React, { useState, useEffect, useCallback, useRef } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  RefreshControl,
  Alert,
  Platform,
  PermissionsAndroid,
  Dimensions,
  ActivityIndicator,
  InteractionManager,
} from "react-native";
import { LinearGradient } from "expo-linear-gradient";
import { Colors } from "@/assets/styles/App.styles";
import { Pedometer } from "expo-sensors";
import {
  MaterialCommunityIcons,
  FontAwesome5,
  Ionicons,
} from "@expo/vector-icons";
import { AnimatedCircularProgress } from "react-native-circular-progress";
import AsyncStorage from "@react-native-async-storage/async-storage";
import dayjs from "dayjs";
import { styles } from "@/assets/styles/App.styles";
import Toolbar from "@/app/components/Toolbar";
import Topbar from "@/app/components/Topbar";
import { useFocusEffect, useRouter } from "expo-router";
import { useUserProfile } from "@/app/hooks/queries/useProfile";
import useHealthStore from "@/app/store/healthStore";
import { useSyncSteps } from "@/app/hooks/mutations/useSyncSteps";
import useAuthStore from "@/app/store/authStore";
import * as stepsEndpoints from "@/app/api/endpoints/steps";
import NetInfo from "@react-native-community/netinfo";
import Toast from "@/app/components/common/ToastMessage";
import TestMode from "@/app/components/TestMode";

// Removed hardcoded keys. We'll generate keys based on the logged in user's id.
const HISTORY_FETCH_INTERVAL = 30 * 60 * 1000; // 30분 간격으로 서버에서 히스토리 fetch

export default function Homepage() {
  const router = useRouter();
  const { data: user, isLoading, error } = useUserProfile();
  const {
    steps,
    dailyGoal,
    startPedometer,
    loadCurrentSteps,
    isPedometerAvailable,
  } = useHealthStore();

  // 누락된 상태 변수 추가
  const [stepHistory, setStepHistory] = useState([]);
  const [isHistoryLoading, setIsHistoryLoading] = useState(false);

  // 성능 최적화를 위한 추가 ref
  const isMountedRef = useRef(true);
  const isInitializedRef = useRef(false);
  const taskQueueRef = useRef([]);
  const timeoutsRef = useRef([]);
  const isFocusedRef = useRef(false);

  // 작업 큐 관리 함수들
  const queueTask = useCallback((task) => {
    if (isMountedRef.current) {
      taskQueueRef.current.push(task);
      processTaskQueue();
    }
  }, []);

  // 작업 큐 처리 함수
  const processTaskQueue = useCallback(() => {
    if (taskQueueRef.current.length === 0 || !isMountedRef.current) return;

    // UI 작업이 완료된 후 작업 처리 시작
    InteractionManager.runAfterInteractions(() => {
      if (!isMountedRef.current) return;

      const task = taskQueueRef.current.shift();
      if (task) {
        // 비동기 작업 실행
        Promise.resolve()
          .then(() => task())
          .catch((error) => {
            // 오류가 발생해도 다음 작업은 계속 진행
            if (isMountedRef.current) {
              console.warn("Task error:", error);
            }
          })
          .finally(() => {
            // 다음 작업이 있으면 계속 처리
            if (isMountedRef.current && taskQueueRef.current.length > 0) {
              processTaskQueue();
            }
          });
      }
    });
  }, []);

  // 안전한 타임아웃 설정 함수
  const safeSetTimeout = useCallback((callback, delay) => {
    if (!isMountedRef.current) return null;

    const timeoutId = setTimeout(() => {
      // 타임아웃 실행 시 자동으로 배열에서 제거
      timeoutsRef.current = timeoutsRef.current.filter(
        (id) => id !== timeoutId
      );

      // 마운트 상태와 포커스 상태 확인 후 콜백 실행
      if (isMountedRef.current && isFocusedRef.current) {
        callback();
      }
    }, delay);

    // 타임아웃 ID를 배열에 저장
    timeoutsRef.current.push(timeoutId);
    return timeoutId;
  }, []);

  // 모든 타임아웃 정리 함수
  const clearAllTimeouts = useCallback(() => {
    if (timeoutsRef.current.length > 0) {
      timeoutsRef.current.forEach((id) => clearTimeout(id));
      timeoutsRef.current = [];
    }
  }, []);

  const initializationPromiseRef = useRef(null);
  const prevUserRef = useRef(null);
  const token = useAuthStore((state) => state.token);
  const {
    mutate: syncSteps,
    isLoading: isSyncing,
    isError: syncError,
    data: syncData,
  } = useSyncSteps();

  // Helper functions to generate user-specific storage keys
  const userId = user?.id;
  const getDailyStepKey = () =>
    userId ? `DAILY_STEP_${userId}` : "DAILY_STEP";
  const getDailyDateKey = () =>
    userId ? `DAILY_DATE_${userId}` : "DAILY_DATE";
  const getStepHistoryKey = () =>
    userId ? `STEP_HISTORY_${userId}` : "STEP_HISTORY";
  const getLastHistoryFetchKey = () =>
    userId ? `LAST_HISTORY_FETCH_${userId}` : "LAST_HISTORY_FETCH";

  const [debouncedSteps, setDebouncedSteps] = useState(0);

  // debounce 로직 개선
  useEffect(() => {
    const timeoutId = safeSetTimeout(() => {
      if (steps > 0 && steps !== debouncedSteps) {
        setDebouncedSteps(steps);
      }
    }, 300);

    return () => {
      if (timeoutId) clearTimeout(timeoutId);
    };
  }, [steps, debouncedSteps, safeSetTimeout]);

  // 걸음 수 저장 로직 최적화
  useEffect(() => {
    if (debouncedSteps > 0) {
      // 메인 스레드 블로킹 방지를 위해 작업 큐에 추가
      queueTask(() =>
        saveStepHistory(dayjs().format("YYYY-MM-DD"), debouncedSteps)
      );
    }
  }, [debouncedSteps, queueTask]);

  // 사용자 정보 변경 감지 최적화
  useEffect(() => {
    if (
      user &&
      token &&
      user.daily_step_goal &&
      (!prevUserRef.current ||
        prevUserRef.current?.id !== user.id ||
        prevUserRef.current?.daily_step_goal !== user.daily_step_goal) &&
      isMountedRef.current
    ) {
      // 중요한 상태 업데이트는 즉시 실행
      useHealthStore.getState().setDailyGoal(Number(user.daily_step_goal));

      // 비용이 큰 작업은 큐에 넣어 UI 블로킹 방지
      queueTask(() => {
        if (isMountedRef.current) {
          loadCurrentSteps();
          prevUserRef.current = user;
        }
      });
    }
  }, [user, token, queueTask]);

  // 네트워크 상태 추적 변수 추가
  const [isConnected, setIsConnected] = useState(true);
  const networkListenerRef = useRef(null);

  // 네트워크 연결 상태 모니터링 및 연결 복구 시 보류 중인 동기화 시도
  useEffect(() => {
    // 초기 연결 상태 확인
    const checkInitialConnection = async () => {
      try {
        const state = await NetInfo.fetch();
        setIsConnected(
          state.isConnected && state.isInternetReachable !== false
        );
      } catch (error) {
        // 네트워크 상태 확인 오류 처리
      }
    };

    // UI 스레드 블로킹 방지
    InteractionManager.runAfterInteractions(() => {
      if (isMountedRef.current) {
        checkInitialConnection();
      }
    });

    // 네트워크 상태 변경 리스너 등록
    networkListenerRef.current = NetInfo.addEventListener((state) => {
      const newConnected =
        state.isConnected && state.isInternetReachable !== false;

      // 상태 변경 감지 시 큐에 넣어 UI 블로킹 방지
      if (isMountedRef.current) {
        setIsConnected(newConnected);

        // 이전 상태가 오프라인에서 온라인으로 변경된 경우
        if (!isConnected && newConnected && Platform.OS === "android") {
          // 보류 중인 걸음수 동기화를 작업 큐에 추가
          queueTask(() => {
            if (isMountedRef.current) {
              return stepsEndpoints.syncPendingSteps();
            }
          });
        }
      }
    });

    return () => {
      if (networkListenerRef.current) {
        networkListenerRef.current();
        networkListenerRef.current = null;
      }
    };
  }, [isConnected, queueTask]);

  // 컴포넌트 마운트/언마운트 생명주기 관리
  useEffect(() => {
    isMountedRef.current = true;

    return () => {
      isMountedRef.current = false;
      isFocusedRef.current = false;

      // 모든 타임아웃과 작업 큐 정리
      clearAllTimeouts();
      taskQueueRef.current = [];

      // 현재 걸음 수 저장
      const currentSteps = useHealthStore.getState().steps;
      AsyncStorage.setItem(getDailyStepKey(), String(currentSteps)).catch(
        () => {}
      );
      AsyncStorage.setItem(
        getDailyDateKey(),
        dayjs().format("YYYY-MM-DD")
      ).catch(() => {});

      isInitializedRef.current = false;
    };
  }, [clearAllTimeouts]);

  // 화면에 포커스될 때 걸음 수 초기화 및 추적 시작
  useFocusEffect(
    useCallback(() => {
      isFocusedRef.current = true;

      // 이미 초기화 작업이 진행 중이라면 중복 실행 방지
      if (initializationPromiseRef.current) {
        return;
      }

      const initializeSteps = async () => {
        try {
          // 이미 초기화되었거나 마운트 해제된 경우 건너뜀
          if (
            isInitializedRef.current ||
            !isMountedRef.current ||
            !isFocusedRef.current
          ) {
            return;
          }

          isInitializedRef.current = true;

          if (Platform.OS === "android") {
            const granted = await PermissionsAndroid.request(
              PermissionsAndroid.PERMISSIONS.ACTIVITY_RECOGNITION
            );
            if (granted !== PermissionsAndroid.RESULTS.GRANTED) {
              Alert.alert(
                "Permission Required",
                "We need activity recognition permission to count your steps."
              );
              return;
            }

            // 안드로이드에서 화면에 포커스가 올 때마다 보류 중인 걸음수 동기화 시도
            if (isConnected && isFocusedRef.current) {
              // 동기화 작업을 큐에 넣어 메인 스레드 블로킹 방지
              queueTask(() => {
                if (isFocusedRef.current) {
                  return stepsEndpoints.syncPendingSteps();
                }
              });
            }
          }

          // AsyncStorage 작업을 병렬로 처리하여 성능 개선
          const [storedSteps, storedDate] = await Promise.all([
            AsyncStorage.getItem(getDailyStepKey()),
            AsyncStorage.getItem(getDailyDateKey()),
          ]);

          const today = dayjs().format("YYYY-MM-DD");

          // 마운트 상태와 포커스 상태 확인
          if (!isMountedRef.current || !isFocusedRef.current) return;

          // 오늘 날짜이고 저장된 값이 존재한다면 글로벌 state에 반영
          if (storedDate === today && storedSteps) {
            useHealthStore.setState({ steps: parseInt(storedSteps, 10) });
          }

          // UI 작업이 완료된 후 무거운 작업 실행
          await InteractionManager.runAfterInteractions();

          if (!isMountedRef.current || !isFocusedRef.current) return;

          await loadCurrentSteps();

          // 기존 구독 중지 후 새로운 구독 시작
          useHealthStore.getState().stopPedometer();
          await startPedometer();

          // 히스토리 로드는 우선 순위가 낮으므로 다른 작업이 완료된 후 실행
          queueTask(() => {
            if (isFocusedRef.current) {
              return loadStepHistory();
            }
          });
        } catch (error) {
          // 초기화 오류 처리
        } finally {
          // 초기화 작업 완료 표시
          initializationPromiseRef.current = null;
        }
      };

      // 초기화 작업 시작 및 추적
      initializationPromiseRef.current = initializeSteps();

      return () => {
        // 화면에서 포커스가 떠날 때
        isFocusedRef.current = false;

        // 현재 걸음 수를 저장하여 데이터 손실 방지
        if (isMountedRef.current) {
          const currentSteps = useHealthStore.getState().steps;

          // 비동기 저장 작업을 큐에 추가하여 UI 블로킹 방지
          Promise.all([
            AsyncStorage.setItem(getDailyStepKey(), String(currentSteps)),
            AsyncStorage.setItem(
              getDailyDateKey(),
              dayjs().format("YYYY-MM-DD")
            ),
          ]).catch(() => {});
        }
      };
    }, [isConnected, queueTask])
  );

  const saveStepHistory = async (date, steps) => {
    try {
      // 로컬 히스토리 데이터를 객체 형태로 로드
      const historyStr = await AsyncStorage.getItem(getStepHistoryKey());
      let history = historyStr ? JSON.parse(historyStr) : {};

      // 기존 형식과 새 형식 모두 처리
      if (typeof history === "object") {
        // 날짜가 이미 존재하는 경우 확인
        const existingEntry = history[date];

        if (!existingEntry || typeof existingEntry === "number") {
          // 단순 숫자 형태로 저장된 경우 새 형식으로 변환
          history[date] = {
            steps: steps,
            calories: Math.round(steps * 0.04),
            distance: Number((steps * 0.0007).toFixed(2)),
            goalAchieved: steps >= dailyGoal,
            date: date,
          };
        } else {
          // 이미 객체 형태인 경우 업데이트
          existingEntry.steps = steps;
          existingEntry.calories = Math.round(steps * 0.04);
          existingEntry.distance = Number((steps * 0.0007).toFixed(2));
          existingEntry.goalAchieved = steps >= dailyGoal;
          history[date] = existingEntry;
        }

        await AsyncStorage.setItem(
          getStepHistoryKey(),
          JSON.stringify(history)
        );

        // 히스토리 상태 업데이트 (UI 즉시 반영)
        const updatedHistoryArray = convertHistoryToArray(history);
        setStepHistory(updatedHistoryArray);

        console.log(`Updated step history for ${date}: ${steps} steps`);
      }
    } catch (error) {
      console.error("Error saving step history:", error);
    }
  };

  // 히스토리 객체를 배열로 변환하는 함수
  const convertHistoryToArray = (historyObj) => {
    const historyArray = Object.keys(historyObj).map((key) => {
      const item = historyObj[key];
      return {
        date: key,
        steps: typeof item === "object" ? item.steps : item,
        percentage:
          typeof dailyGoal === "number"
            ? Math.min(
                100,
                Math.round(
                  ((typeof item === "object" ? item.steps : item) / dailyGoal) *
                    100
                )
              )
            : 0,
        calories:
          typeof item === "object"
            ? item.calories
            : Math.round((typeof item === "object" ? item.steps : item) * 0.04),
        distance:
          typeof item === "object"
            ? item.distance
            : Number(
                (
                  (typeof item === "object" ? item.steps : item) * 0.0007
                ).toFixed(2)
              ),
        goalAchieved:
          typeof item === "object"
            ? item.goalAchieved
            : typeof item === "number" && dailyGoal
            ? item >= dailyGoal
            : false,
      };
    });

    // 최신 날짜 순(내림차순) 정렬
    return historyArray.sort((a, b) => {
      return dayjs(b.date).diff(dayjs(a.date));
    });
  };

  const loadStepHistory = async () => {
    try {
      setIsHistoryLoading(true);

      // 마지막 히스토리 fetch 시간 확인
      const lastFetchStr = await AsyncStorage.getItem(getLastHistoryFetchKey());
      const now = Date.now();
      const shouldFetchFromServer =
        !lastFetchStr || now - parseInt(lastFetchStr) > HISTORY_FETCH_INTERVAL;

      // 먼저 로컬 히스토리 로드 (표시 지연 방지)
      await loadLocalStepHistory();

      // 네트워크 상태 확인
      const isConnected = await checkNetwork();

      if (!isConnected) {
        console.log("No network connection, using local step history only");
        setIsHistoryLoading(false);
        return;
      }

      // 서버에서 fetch할 필요가 없는 경우
      if (!shouldFetchFromServer) {
        console.log("Skipping server fetch, using recent local history");
        setIsHistoryLoading(false);
        return;
      }

      // 서버에서 전체 걸음 히스토리를 가져옴
      try {
        const response = await stepsEndpoints.getAllSteps();
        console.log(
          "Loaded step history from API:",
          response.length,
          "records"
        );

        // 응답이 배열인지 확인
        if (!Array.isArray(response)) {
          console.error("API response is not an array:", response);
          setIsHistoryLoading(false);
          return;
        }

        // 마지막 fetch 시간 업데이트
        await AsyncStorage.setItem(getLastHistoryFetchKey(), now.toString());

        // 응답은 예시와 같이 배열 형태임
        const historyArray = response.map((item) => ({
          date: item.record_date,
          steps: item.steps_count,
          points: item.points_earned || 0,
          calories: item.calories_burned || 0,
          distance: item.distance_km || 0,
          // dailyGoal가 숫자일 때 백분율 계산
          percentage:
            typeof dailyGoal === "number"
              ? Math.min(100, Math.round((item.steps_count / dailyGoal) * 100))
              : 0,
        }));

        // 최신 날짜 순(내림차순) 정렬
        historyArray.sort((a, b) => dayjs(b.date).diff(dayjs(a.date)));
        setStepHistory(historyArray);

        // 로컬 스토리지에도 저장
        saveHistoryToLocal(historyArray);

        // 성공 메시지 표시 (선택적)
        // ToastMessage("성공", "걸음수 기록이 업데이트되었습니다", "success");
      } catch (error) {
        console.error("Error loading step history from API:", error);
        // ToastMessage("오류", "걸음수 기록 로드 실패", "danger");
      } finally {
        setIsHistoryLoading(false);
      }
    } catch (error) {
      console.error("Error in loadStepHistory:", error);
      setIsHistoryLoading(false);
    }
  };

  // 로컬에 저장된 히스토리 불러오기
  const loadLocalStepHistory = async () => {
    try {
      const historyStr = await AsyncStorage.getItem(getStepHistoryKey());
      if (!historyStr) {
        console.log("No local step history found");
        setStepHistory([]);
        return;
      }

      const historyData = JSON.parse(historyStr);

      // 히스토리 배열로 변환
      const historyArray = convertHistoryToArray(historyData);
      setStepHistory(historyArray);

      console.log("Loaded local step history:", historyArray.length, "records");
    } catch (error) {
      console.error("Error loading local step history:", error);
      setStepHistory([]);
    }
  };

  // 히스토리 데이터를 로컬에 저장
  const saveHistoryToLocal = async (historyArray) => {
    try {
      const historyObj = {};

      // 배열을 객체 형태로 변환하여 저장
      historyArray.forEach((item) => {
        historyObj[item.date] = {
          steps: item.steps,
          calories: item.calories || 0,
          distance: item.distance || 0,
          goalAchieved: item.percentage >= 100,
          date: item.date,
        };
      });

      await AsyncStorage.setItem(
        getStepHistoryKey(),
        JSON.stringify(historyObj)
      );
      console.log("Saved step history to local storage");
    } catch (error) {
      console.error("Error saving step history to local:", error);
    }
  };

  // 네트워크 상태 확인 함수
  const checkNetwork = async () => {
    try {
      // NetInfo가 이미 import 되어 있어야 함
      const state = await NetInfo.fetch();
      return state.isConnected && state.isInternetReachable;
    } catch (error) {
      console.error("Error checking network:", error);
      return false;
    }
  };

  const handleStepCount = async (newSteps) => {
    try {
      const updatedSteps = steps + newSteps;
      startPedometer();
      await AsyncStorage.setItem(getDailyStepKey(), updatedSteps.toString());

      // Save to history
      const today = dayjs().format("YYYY-MM-DD");
      await saveStepHistory(today, updatedSteps);
    } catch (error) {
      console.error("Error handling step count:", error);
    }
  };

  const handleSync = useCallback(() => {
    // Add a console log to see when it's triggered
    console.log("Manual sync triggered with steps:", steps);

    // We need to prevent sync calls with the same steps value
    if (!steps || typeof steps !== "number") {
      console.log("Skipping sync - invalid steps value:", steps);
      return;
    }

    // Add a lastSyncedSteps check to avoid repeated syncs with same value
    const lastSyncedSteps = useHealthStore.getState().lastSyncedSteps;
    if (steps <= lastSyncedSteps) {
      console.log(
        `Skipping sync - already synced steps (${steps} <= ${lastSyncedSteps})`
      );
      return;
    }

    // 네트워크 연결 확인 후 동기화
    if (isConnected) {
      syncSteps({ steps, dailyGoal });
    } else if (Platform.OS === "android") {
      // 오프라인 상태인 경우 안드로이드에서 로컬 저장
      console.log(
        "[ANDROID] 오프라인 상태 - 걸음수를 로컬에 임시 저장:",
        steps
      );
      AsyncStorage.getItem("PENDING_STEPS_SYNC")
        .then((stepsJSON) => {
          const pendingSteps = stepsJSON ? JSON.parse(stepsJSON) : [];
          pendingSteps.push({
            steps,
            timestamp: Date.now(),
            date: dayjs().format("YYYY-MM-DD"),
          });
          return AsyncStorage.setItem(
            "PENDING_STEPS_SYNC",
            JSON.stringify(pendingSteps.slice(-10))
          );
        })
        .catch((error) => {
          console.error("[ANDROID] 오프라인 걸음수 임시 저장 오류:", error);
        });
    }
  }, [steps, dailyGoal, isConnected]);

  // 데이터 동기화 useEffect 수정
  useEffect(() => {
    // 걸음수 동기화 주기를 15분에서 30분으로 늘리고
    // 디바운스 형태로 구현하여 불필요한 API 호출 최소화
    let syncTimeout;

    const debouncedSync = () => {
      if (syncTimeout) {
        clearTimeout(syncTimeout);
      }

      // 5초 후에 실행되도록 디바운싱
      syncTimeout = setTimeout(() => {
        if (steps > 0) {
          handleSync();
        }
      }, 5000);
    };

    // 걸음수가 변경될 때만 디바운스 동기화 호출
    if (steps > 0) {
      debouncedSync();
    }

    // 정기적인 동기화 실행 (30분마다) - 단 상태가 실제로 변경된 경우만
    const interval = setInterval(() => {
      const currentState = useHealthStore.getState();
      // 마지막 동기화된 걸음 수와 현재 걸음 수를 비교하여 변경된 경우만 동기화
      if (currentState.steps > currentState.lastSyncedSteps) {
        handleSync();
      }
    }, 30 * 60 * 1000);

    return () => {
      clearInterval(interval);
      if (syncTimeout) {
        clearTimeout(syncTimeout);
      }
    };
  }, [steps, handleSync]);

  const [showAllHistory, setShowAllHistory] = useState(false);

  // 표시할 히스토리 데이터 계산
  const displayedHistory = showAllHistory
    ? stepHistory
    : stepHistory.slice(0, 5);

  if (isLoading) {
    return (
      <View style={{ flex: 1, justifyContent: "center", alignItems: "center" }}>
        <ActivityIndicator size="large" color={Colors.primary} />
      </View>
    );
  }

  if (error) {
    return (
      <View
        style={{
          flex: 1,
          justifyContent: "center",
          alignItems: "center",
          padding: 20,
        }}
      >
        <Text
          style={{
            color: Colors.danger,
            fontFamily: "montbold",
            textAlign: "center",
          }}
        >
          Error: {error.message}
        </Text>
        <TouchableOpacity
          style={{
            marginTop: 20,
            backgroundColor: Colors.primary,
            padding: 10,
            borderRadius: 10,
          }}
          onPress={() => router.replace("/")}
        >
          <Text style={{ color: Colors.white, fontFamily: "montmedium" }}>
            Yeniden dene
          </Text>
        </TouchableOpacity>
      </View>
    );
  }

  return (
    <View style={{ flex: 1, backgroundColor: Colors.white }}>
      <Topbar title="Anasayfa" />
      {/* {__DEV__ && <TestMode />} */}

      <ScrollView
        contentContainerStyle={{ paddingBottom: 100 }}
        refreshControl={
          <RefreshControl
            refreshing={isHistoryLoading}
            onRefresh={loadStepHistory}
            colors={[Colors.primary]}
          />
        }
      >
        <LinearGradient
          colors={[
            Colors.gradientLight[0],
            Colors.gradientLight[1],
            Colors.gradientLight[2],
          ]}
          style={{
            padding: 20,
            borderBottomLeftRadius: 30,
            borderBottomRightRadius: 30,
            paddingBottom: 40,
          }}
        >
          <View style={{ alignItems: "center", marginTop: 20 }}>
            <AnimatedCircularProgress
              size={200}
              width={15}
              fill={(steps / dailyGoal) * 100}
              tintColor={Colors.primary}
              backgroundColor={Colors.lightGray}
              rotation={0}
              lineCap="round"
            >
              {() => (
                <View style={{ alignItems: "center" }}>
                  <Text
                    style={{
                      fontFamily: "montbold",
                      fontSize: 40,
                      color: Colors.primary,
                    }}
                  >
                    {steps}
                  </Text>
                  <Text
                    style={{ fontFamily: "montmedium", color: Colors.primary }}
                  >
                    Adım
                  </Text>
                </View>
              )}
            </AnimatedCircularProgress>
          </View>

          <View
            style={{
              flexDirection: "row",
              justifyContent: "space-between",
              marginTop: 30,
            }}
          >
            <View
              style={{
                backgroundColor: Colors.white,
                padding: 20,
                borderRadius: 15,
                flex: 1,
                marginRight: 10,
                alignItems: "center",
                shadowColor: Colors.tertiary,
                shadowOffset: { width: 0, height: 2 },
                shadowOpacity: 0.1,
                shadowRadius: 4,
                elevation: 3,
              }}
            >
              <FontAwesome5 name="bullseye" size={24} color={Colors.primary} />
              <Text
                style={{
                  fontFamily: "montbold",
                  fontSize: 24,
                  color: Colors.primary,
                  marginTop: 10,
                }}
              >
                {dailyGoal}
              </Text>
              <Text
                style={{
                  fontFamily: "montregular",
                  fontSize: 14,
                  color: Colors.tertiary,
                }}
              >
                Günlük Hedef
              </Text>
            </View>

            <View
              style={{
                backgroundColor: Colors.white,
                padding: 20,
                borderRadius: 15,
                flex: 1,
                marginLeft: 10,
                alignItems: "center",
                shadowColor: Colors.tertiary,
                shadowOffset: { width: 0, height: 2 },
                shadowOpacity: 0.1,
                shadowRadius: 4,
                elevation: 3,
              }}
            >
              <MaterialCommunityIcons
                name="percent"
                size={24}
                color={Colors.primary}
              />
              <Text
                style={{
                  fontFamily: "montbold",
                  fontSize: 24,
                  color: Colors.primary,
                  marginTop: 10,
                }}
              >
                {Math.min(100, Math.round((steps / dailyGoal) * 100))}%
              </Text>
              <Text
                style={{
                  fontFamily: "montregular",
                  fontSize: 14,
                  color: Colors.tertiary,
                }}
              >
                Tamamlanan
              </Text>
            </View>
          </View>
        </LinearGradient>

        <View
          style={{
            flexDirection: "row",
            flexWrap: "wrap",
            gap: 10,
            padding: 20,
          }}
        >
          <TouchableOpacity
            onPress={() => {
              console.log("Navigating to Tournaments.");
              router.push("pages/home/tournaments");
            }}
            style={{
              backgroundColor: Colors.white,
              padding: 15,
              borderRadius: 15,
              width: "48%",
              alignItems: "center",
              shadowColor: Colors.tertiary,
              shadowOffset: { width: 0, height: 2 },
              shadowOpacity: 0.1,
              shadowRadius: 4,
              elevation: 3,
            }}
          >
            <MaterialCommunityIcons
              name="trophy"
              size={32}
              color={Colors.primary}
            />
            <Text
              style={{
                fontFamily: "montmedium",
                fontSize: 16,
                color: Colors.tertiary,
                marginTop: 5,
              }}
            >
              Yarışmalar
            </Text>
          </TouchableOpacity>

          <TouchableOpacity
            onPress={() => {
              console.log("Navigating to Wallet.");
              router.push("pages/home/wallet");
            }}
            style={{
              backgroundColor: Colors.white,
              padding: 15,
              borderRadius: 15,
              width: "48%",
              alignItems: "center",
              shadowColor: Colors.tertiary,
              shadowOffset: { width: 0, height: 2 },
              shadowOpacity: 0.1,
              shadowRadius: 4,
              elevation: 3,
            }}
          >
            <Ionicons name="wallet" size={32} color={Colors.primary} />
            <Text
              style={{
                fontFamily: "montmedium",
                fontSize: 16,
                color: Colors.tertiary,
                marginTop: 5,
              }}
            >
              Cüzdan
            </Text>
          </TouchableOpacity>
        </View>

        <TouchableOpacity
          activeOpacity={0.6}
          onPress={() => {
            router.push("pages/home/convertStepsToPoints");
          }}
          style={{
            marginTop: 10,
            width: "90%",
            alignItems: "center",
            justifyContent: "center",
            backgroundColor: Colors.primary,
            padding: 10,
            borderRadius: 10,
            alignSelf: "center",
          }}
        >
          <Text
            style={{
              fontFamily: "montmedium",
              fontSize: 14,
              color: Colors.white,
            }}
          >
            Adımlarını Puana Dönüştür
          </Text>
        </TouchableOpacity>

        <View style={{ padding: 20 }}>
          <Text
            style={{
              fontFamily: "montbold",
              fontSize: 18,
              color: Colors.tertiary,
              marginBottom: 10,
            }}
          >
            Adım Geçmişi
          </Text>

          {isHistoryLoading && stepHistory.length === 0 ? (
            <View style={{ alignItems: "center", padding: 20 }}>
              <ActivityIndicator size="small" color={Colors.primary} />
              <Text
                style={{
                  marginTop: 10,
                  fontFamily: "montregular",
                  color: Colors.tertiary,
                }}
              >
                Adım geçmişi yükleniyor...
              </Text>
            </View>
          ) : stepHistory.length === 0 ? (
            <View
              style={{
                backgroundColor: Colors.white,
                padding: 20,
                borderRadius: 10,
                alignItems: "center",
                shadowColor: Colors.tertiary,
                shadowOffset: { width: 0, height: 2 },
                shadowOpacity: 0.1,
                shadowRadius: 4,
                elevation: 3,
              }}
            >
              <Text
                style={{ fontFamily: "montmedium", color: Colors.tertiary }}
              >
                Henüz adım geçmişiniz bulunmuyor.
              </Text>
            </View>
          ) : (
            displayedHistory.map((item, index) => (
              <View
                key={index}
                style={{
                  backgroundColor: Colors.white,
                  padding: 15,
                  borderRadius: 10,
                  marginBottom: 10,
                  flexDirection: "row",
                  justifyContent: "space-between",
                  alignItems: "center",
                  shadowColor: Colors.tertiary,
                  shadowOffset: { width: 0, height: 2 },
                  shadowOpacity: 0.1,
                  shadowRadius: 4,
                  elevation: 3,
                }}
              >
                <View>
                  <Text
                    style={{ fontFamily: "montbold", color: Colors.tertiary }}
                  >
                    {dayjs(item.date).format("DD MMMM YYYY")}
                  </Text>
                  <Text
                    style={{ fontFamily: "montmedium", color: Colors.primary }}
                  >
                    {item.steps} Adım
                  </Text>
                  {item.calories > 0 && (
                    <Text
                      style={{
                        fontFamily: "montregular",
                        color: Colors.tertiary,
                        fontSize: 12,
                      }}
                    >
                      {item.calories} Kalori • {item.distance} km
                    </Text>
                  )}
                </View>
                <Text
                  style={{
                    fontFamily: "montbold",
                    color: Colors.primary,
                  }}
                >
                  {item.percentage}%
                </Text>
              </View>
            ))
          )}

          {!isHistoryLoading && stepHistory.length > 5 && (
            <TouchableOpacity
              onPress={() => setShowAllHistory(!showAllHistory)}
              style={{
                alignItems: "center",
                marginTop: 10,
                padding: 10,
              }}
            >
              <Text style={{ fontFamily: "montmedium", color: Colors.primary }}>
                {showAllHistory ? "Kısa Görüntüle" : "Daha Fazla Göster"}
              </Text>
            </TouchableOpacity>
          )}
        </View>
      </ScrollView>
      <Toolbar selectedMenu={1} />
    </View>
  );
}
