import React, { useState, useEffect, useRef } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  RefreshControl,
  Dimensions,
  Platform,
  Alert,
  PermissionsAndroid,
} from "react-native";
import { LinearGradient } from "expo-linear-gradient";
import { Colors } from "@/assets/styles/App.styles";
import { Pedometer } from "expo-sensors";
import {
  MaterialCommunityIcons,
  FontAwesome5,
  Ionicons,
} from "@expo/vector-icons";
import { AnimatedCircularProgress } from "react-native-circular-progress";
import AsyncStorage from "@react-native-async-storage/async-storage";

import { LineChart } from "react-native-chart-kit";
import dayjs from "dayjs";
import { styles } from "@/assets/styles/App.styles";
import Toolbar from "@/app/components/Toolbar";
import Topbar from "@/app/components/Topbar";
import { useRouter } from "expo-router";

const DEFAULT_USER = {
  name: "Ahmet",
  surname: "Yılmaz",
  goal: 10000,
  avatar: "https://i.pravatar.cc/200",
  email: "ahmet@yilmaz.com",
  phone: "5551234567",
  country: "TR",
  countryMultiplier: 1.0,
  earnedPoints: 0,
  inviteCount: 0,
  referralCode: "123456",
};

const STEP_HISTORY_KEY = "STEP_HISTORY";
const DAILY_STEP_KEY = "DAILY_STEP";
const DAILY_DATE_KEY = "DAILY_DATE";

export default function HomePageOld() {
  const [steps, setSteps] = useState(0);
  const [points, setPoints] = useState(0);
  const [dailyGoal, setDailyGoal] = useState(6000);
  const [stepHistory, setStepHistory] = useState([]);
  const [weeklyStats, setWeeklyStats] = useState({
    labels: ["Pzt", "Sal", "Çar", "Per", "Cum", "Cmt", "Paz"],
    datasets: [
      {
        data: [0, 0, 0, 0, 0, 0, 0],
        color: (opacity = 1) => Colors.primary,
        strokeWidth: 2,
      },
    ],
  });
  const [countryMultiplier, setCountryMultiplier] = useState(1.0); // Bu değer backend'den gelecek
  const screenWidth = Dimensions.get("window").width;
  const router = useRouter();
  const [user, setUser] = useState(DEFAULT_USER);
  const [isPedometerAvailable, setIsPedometerAvailable] = useState("checking");
  const subscription = useRef(null);

  useEffect(() => {
    let isMounted = true; // To prevent state updates if component is unmounted

    const requestPedometerPermission = async () => {
      if (Platform.OS === "android") {
        try {
          console.log("Requesting Activity Recognition permission...");
          const granted = await PermissionsAndroid.request(
            PermissionsAndroid.PERMISSIONS.ACTIVITY_RECOGNITION,
            {
              title: "Activity Recognition Permission",
              message:
                "This app needs access to your physical activity to track steps.",
              buttonNeutral: "Ask Me Later",
              buttonNegative: "Cancel",
              buttonPositive: "OK",
            }
          );
          if (granted === PermissionsAndroid.RESULTS.GRANTED) {
            console.log("Activity Recognition permission granted");
            return true;
          } else {
            console.log("Activity Recognition permission denied");
            Alert.alert(
              "Permission Denied",
              "Activity Recognition permission is required to track steps.",
              [{ text: "OK" }]
            );
            return false;
          }
        } catch (err) {
          console.warn(err);
          return false;
        }
      } else {
        // iOS permissions are handled via Info.plist (NSMotionUsageDescription)
        return true;
      }
    };

    const initializePedometer = async () => {
      console.log("Initializing Pedometer...");
      try {
        const permission = await requestPedometerPermission();
        if (!permission) {
          console.log(
            "Permission not granted. Exiting pedometer initialization."
          );
          return;
        }

        const available = await Pedometer.isAvailableAsync();
        console.log("Pedometer availability:", available);
        if (isMounted) {
          setIsPedometerAvailable(String(available));
        }

        if (available) {
          console.log(
            "Pedometer is available. Proceeding to load step history and subscribe."
          );
          // Load step history
          await loadStepHistory();

          // Load today's step count
          const today = dayjs().format("YYYY-MM-DD");
          const storedDate = await AsyncStorage.getItem(DAILY_DATE_KEY);
          console.log(`Stored Date: ${storedDate}, Today: ${today}`);
          let storedSteps = 0;

          if (storedDate === today) {
            const stepsCount = await AsyncStorage.getItem(DAILY_STEP_KEY);
            storedSteps = stepsCount ? parseInt(stepsCount, 10) : 0;
            console.log(`Loaded steps for today: ${storedSteps}`);
          } else {
            // New day, reset step count
            await AsyncStorage.setItem(DAILY_STEP_KEY, "0");
            await AsyncStorage.setItem(DAILY_DATE_KEY, today);
            storedSteps = 0;
            console.log("New day detected. Step count reset to 0.");
            // Optionally, you can push the previous day's steps to history here
          }

          if (isMounted) {
            setSteps(storedSteps);
            setPoints(storedSteps * user.countryMultiplier);
            console.log(
              `Initial steps set to: ${storedSteps}, Points set to: ${
                storedSteps * user.countryMultiplier
              }`
            );
          }

          // Subscribe to step count updates
          subscription.current = Pedometer.watchStepCount((result) => {
            console.log("Received step count update:", result.steps);
            handleStepCount(result.steps);
          });
          console.log("Subscribed to Pedometer watchStepCount.");

          // Optionally, handle daily reset
          scheduleDailyReset();
        } else {
          console.log("Pedometer is not available on this device.");
          Alert.alert(
            "Pedometer Unavailable",
            "Pedometer is not available on this device.",
            [{ text: "OK" }]
          );
        }
      } catch (error) {
        console.error("Error initializing pedometer:", error);
      }
    };

    initializePedometer();

    return () => {
      isMounted = false;
      if (subscription.current) {
        subscription.current.remove();
        console.log("Pedometer subscription removed.");
      }
    };
  }, []);

  useEffect(() => {
    // Update weekly stats whenever stepHistory changes
    console.log("Step history updated. Updating weekly stats.");
    updateWeeklyStats(stepHistory);
  }, [stepHistory]);

  const handleStepCount = async (newSteps) => {
    console.log("Handling new steps:", newSteps);
    try {
      const today = dayjs().format("YYYY-MM-DD");
      const currentSteps = await AsyncStorage.getItem(DAILY_STEP_KEY);
      const updatedSteps = currentSteps
        ? parseInt(currentSteps, 10) + newSteps
        : newSteps;
      console.log(
        `Current Steps: ${currentSteps}, Updated Steps: ${updatedSteps}`
      );

      setSteps((prev) => {
        const newTotal = prev + newSteps;
        console.log(`Steps before update: ${prev}, after update: ${newTotal}`);
        return newTotal;
      });

      setPoints((prev) => {
        const newPoints = prev + newSteps * user.countryMultiplier;
        console.log(
          `Points before update: ${prev}, after update: ${newPoints}`
        );
        return newPoints;
      });

      await AsyncStorage.setItem(DAILY_STEP_KEY, updatedSteps.toString());
      console.log(`Daily steps updated in AsyncStorage: ${updatedSteps}`);

      // Update step history
      await updateStepHistory(updatedSteps, today);
    } catch (error) {
      console.error("Error handling step count:", error);
    }
  };

  const updateStepHistory = async (todaySteps, date) => {
    console.log("Updating step history...");
    try {
      const today = date || dayjs().format("YYYY-MM-DD");
      console.log(`Date for step history update: ${today}`);
      let updatedHistory = [...stepHistory];
      const existingEntryIndex = updatedHistory.findIndex(
        (entry) => entry.date === today
      );

      if (existingEntryIndex >= 0) {
        console.log(`Existing entry found for ${today}. Updating steps.`);
        updatedHistory[existingEntryIndex].steps = todaySteps;
      } else {
        console.log(`No existing entry for ${today}. Adding new entry.`);
        updatedHistory.push({ date: today, steps: todaySteps });
      }

      // Keep only the last 7 days
      if (updatedHistory.length > 7) {
        console.log("Step history exceeds 7 days. Trimming the history.");
        updatedHistory = updatedHistory.slice(updatedHistory.length - 7);
      }

      setStepHistory(updatedHistory);
      console.log("Step history state updated:", updatedHistory);
      await AsyncStorage.setItem(
        STEP_HISTORY_KEY,
        JSON.stringify(updatedHistory)
      );
      console.log("Step history saved to AsyncStorage.");
    } catch (error) {
      console.error("Error updating step history:", error);
    }
  };

  const loadStepHistory = async () => {
    console.log("Loading step history from AsyncStorage...");
    try {
      const history = await AsyncStorage.getItem(STEP_HISTORY_KEY);
      if (history !== null) {
        const parsedHistory = JSON.parse(history);
        setStepHistory(parsedHistory);
        console.log("Step history loaded:", parsedHistory);
      } else {
        console.log("No step history found in AsyncStorage.");
      }
      // Load today's steps
      const today = dayjs().format("YYYY-MM-DD");
      const storedSteps = await AsyncStorage.getItem(DAILY_STEP_KEY);
      const storedDate = await AsyncStorage.getItem(DAILY_DATE_KEY);

      console.log(
        `Stored Steps: ${storedSteps}, Stored Date: ${storedDate}, Today: ${today}`
      );

      if (storedDate !== today) {
        await AsyncStorage.setItem(DAILY_STEP_KEY, "0");
        await AsyncStorage.setItem(DAILY_DATE_KEY, today);
        setSteps(0);
        setPoints(0);
        console.log("New day detected during load. Step count reset to 0.");
      } else {
        const stepsCount = storedSteps ? parseInt(storedSteps, 10) : 0;
        setSteps(stepsCount);
        setPoints(stepsCount * user.countryMultiplier);
        console.log(
          `Steps loaded for today: ${stepsCount}, Points: ${
            stepsCount * user.countryMultiplier
          }`
        );
      }
    } catch (error) {
      console.error("Error loading step history:", error);
    }
  };

  const updateWeeklyStats = (history) => {
    console.log("Updating weekly statistics...");
    const labels = ["Pzt", "Sal", "Çar", "Per", "Cum", "Cmt", "Paz"];
    const today = dayjs();
    const data = [];

    for (let i = 6; i >= 0; i--) {
      const day = today.subtract(i, "day").format("YYYY-MM-DD");
      const entry = history.find((e) => e.date === day);
      data.push(entry ? entry.steps : 0);
      console.log(`Day: ${day}, Steps: ${entry ? entry.steps : 0}`);
    }

    setWeeklyStats({
      labels,
      datasets: [
        {
          data,
          color: (opacity = 1) => Colors.primary,
          strokeWidth: 2,
        },
      ],
    });
    console.log("Weekly stats updated:", {
      labels,
      datasets: [
        {
          data,
          color: Colors.primary,
          strokeWidth: 2,
        },
      ],
    });
  };

  const loadWeeklyStats = async () => {
    // Optional: Implement if you want to preload weekly stats
  };

  const scheduleDailyReset = () => {
    console.log("Scheduling daily reset...");
    const now = dayjs();
    const midnight = dayjs().endOf("day").add(1, "second");
    const millisUntilMidnight = midnight.diff(now, "millisecond");
    console.log(`Milliseconds until midnight: ${millisUntilMidnight}`);

    setTimeout(async () => {
      console.log("Midnight reached. Resetting daily steps.");
      try {
        const today = dayjs().format("YYYY-MM-DD");
        const storedSteps = await AsyncStorage.getItem(DAILY_STEP_KEY);
        const stepsCount = storedSteps ? parseInt(storedSteps, 10) : 0;
        console.log(`Stored Steps before reset: ${stepsCount}`);

        // Update step history with yesterday's steps
        await updateStepHistory(
          stepsCount,
          dayjs().subtract(1, "day").format("YYYY-MM-DD")
        );

        // Reset daily steps
        await AsyncStorage.setItem(DAILY_STEP_KEY, "0");
        await AsyncStorage.setItem(DAILY_DATE_KEY, today);
        setSteps(0);
        setPoints(0);
        console.log("Daily steps reset in AsyncStorage.");
        console.log("Steps and points state reset to 0.");
      } catch (error) {
        console.error("Error during daily reset:", error);
      }
      // Schedule the next reset
      scheduleDailyReset();
    }, millisUntilMidnight);
  };

  return (
    <View style={{ flex: 1 }}>
      <Topbar title={"Anasayfa"} />

      <ScrollView
        contentContainerStyle={{ paddingBottom: 100 }}
        refreshControl={
          <RefreshControl
            refreshing={false}
            onRefresh={() => {
              console.log("Pull-to-refresh triggered.");
              // Implement your refresh logic here
            }}
          />
        }
      >
        <LinearGradient
          colors={[
            Colors.gradientLight[0],
            Colors.gradientLight[1],
            Colors.gradientLight[2],
          ]}
          style={{
            padding: 20,
            borderBottomLeftRadius: 30,
            borderBottomRightRadius: 30,
            paddingBottom: 40,
          }}
        >
          <View style={{ alignItems: "center", marginTop: 20 }}>
            <AnimatedCircularProgress
              size={150}
              width={15}
              fill={(steps / dailyGoal) * 100}
              tintColor={Colors.primary}
              backgroundColor={Colors.lightGray}
              rotation={0}
              lineCap="round"
              duration={500}
              onAnimationComplete={() => console.log("Animation complete")}
            >
              {() => (
                <View style={{ alignItems: "center" }}>
                  <Text
                    style={{
                      fontFamily: "montbold",
                      fontSize: 32,
                      color: Colors.primary,
                    }}
                  >
                    {steps}
                  </Text>
                  <Text
                    style={{ fontFamily: "montmedium", color: Colors.primary }}
                  >
                    Adım
                  </Text>
                </View>
              )}
            </AnimatedCircularProgress>
          </View>

          {/* 
                    Uncomment the following section if you want to display points and country multiplier.
                    */}
          {/*
                    <View style={{ flexDirection: 'row', justifyContent: 'space-between', marginTop: 20 }}>
                        <View style={{
                            backgroundColor: Colors.white,
                            padding: 15,
                            borderRadius: 15,
                            flex: 1,
                            marginRight: 10,
                            shadowColor: Colors.tertiary,
                            shadowOffset: { width: 0, height: 2 },
                            shadowOpacity: 0.1,
                            shadowRadius: 4,
                            elevation: 3,
                        }}>
                            <MaterialCommunityIcons name="star-circle" size={24} color={Colors.primary} />
                            <Text style={{ fontFamily: 'montbold', fontSize: 20, color: Colors.primary, marginTop: 5 }}>
                                {points}
                            </Text>
                            <Text style={{ fontFamily: 'montregular', fontSize: 14, color: Colors.tertiary }}>
                                Adım Puan
                            </Text>
                        </View>

                        <View style={{
                            backgroundColor: Colors.white,
                            padding: 15,
                            borderRadius: 15,
                            flex: 1,
                            marginLeft: 10,
                            shadowColor: Colors.tertiary,
                            shadowOffset: { width: 0, height: 2 },
                            shadowOpacity: 0.1,
                            shadowRadius: 4,
                            elevation: 3,
                        }}>
                            <FontAwesome5 name="globe-americas" size={24} color={Colors.primary} />
                            <Text style={{ fontFamily: 'montbold', fontSize: 20, color: Colors.primary, marginTop: 5 }}>
                                {countryMultiplier}x
                            </Text>
                            <Text style={{ fontFamily: 'montregular', fontSize: 14, color: Colors.tertiary }}>
                                Ülke Çarpanı
                            </Text>
                        </View>
                    </View>
                    */}
        </LinearGradient>

        <View style={{ padding: 20 }}>
          {/* 
                    Uncomment the following section if you want to display quick actions.
                    */}
          {/* <Text style={{ fontFamily: 'montbold', fontSize: 18, color: Colors.tertiary, marginBottom: 15 }}>
                        Hızlı İşlemler
                    </Text> */}

          <View style={{ flexDirection: "row", flexWrap: "wrap", gap: 10 }}>
            <TouchableOpacity
              onPress={() => {
                console.log("Navigating to Tournaments.");
                router.push("pages/home/tournaments");
              }}
              style={{
                backgroundColor: Colors.white,
                padding: 15,
                borderRadius: 15,
                width: "48%",
                alignItems: "center",
                shadowColor: Colors.tertiary,
                shadowOffset: { width: 0, height: 2 },
                shadowOpacity: 0.1,
                shadowRadius: 4,
                elevation: 3,
              }}
            >
              <MaterialCommunityIcons
                name="trophy"
                size={32}
                color={Colors.primary}
              />
              <Text
                style={{
                  fontFamily: "montmedium",
                  fontSize: 16,
                  color: Colors.tertiary,
                  marginTop: 5,
                }}
              >
                Yarışmalar
              </Text>
            </TouchableOpacity>

            <TouchableOpacity
              onPress={() => {
                console.log("Navigating to Wallet.");
                router.push("pages/home/wallet");
              }}
              style={{
                backgroundColor: Colors.white,
                padding: 15,
                borderRadius: 15,
                width: "48%",
                alignItems: "center",
                shadowColor: Colors.tertiary,
                shadowOffset: { width: 0, height: 2 },
                shadowOpacity: 0.1,
                shadowRadius: 4,
                elevation: 3,
              }}
            >
              <Ionicons name="wallet" size={32} color={Colors.primary} />
              <Text
                style={{
                  fontFamily: "montmedium",
                  fontSize: 16,
                  color: Colors.tertiary,
                  marginTop: 5,
                }}
              >
                Cüzdan
              </Text>
            </TouchableOpacity>
          </View>
        </View>

        <View style={{}}>
          <Text
            style={{
              fontFamily: "montbold",
              fontSize: 18,
              color: Colors.tertiary,
              marginBottom: 15,
              marginLeft: 15,
            }}
          >
            Haftalık İstatistikler
          </Text>

          <LineChart
            data={weeklyStats}
            width={screenWidth - 32}
            height={220}
            chartConfig={{
              backgroundColor: Colors.white,
              backgroundGradientFrom: Colors.white,
              backgroundGradientTo: Colors.white,
              decimalPlaces: 0,
              color: (opacity = 1) => Colors.primary,
              style: {
                borderRadius: 16,
              },
              propsForDots: {
                r: "4",
                strokeWidth: "2",
                stroke: Colors.primary,
              },
              propsForBackgroundLines: {
                strokeWidth: 1,
                stroke: Colors.lightGray,
              },
              paddingRight: 24,
              paddingLeft: 0,
            }}
            bezier
            style={{
              marginVertical: 8,
              borderRadius: 16,
            }}
          />

          <View style={{ marginTop: 20, paddingHorizontal: 15 }}>
            <Text
              style={{
                fontFamily: "montbold",
                fontSize: 18,
                color: Colors.tertiary,
                marginBottom: 15,
              }}
            >
              Adım Geçmişi
            </Text>
            {stepHistory
              .slice(-5)
              .reverse()
              .map((day, index) => (
                <View key={index} style={styles.historyItem}>
                  <Text
                    style={{ fontFamily: "montmedium", color: Colors.tertiary }}
                  >
                    {dayjs(day.date).format("DD MMMM YYYY")}
                  </Text>
                  <Text
                    style={{ fontFamily: "montbold", color: Colors.primary }}
                  >
                    {day.steps} Adım
                  </Text>
                </View>
              ))}
          </View>
        </View>
      </ScrollView>
      <Toolbar selectedMenu={1} />
    </View>
  );
}
