import React, { useState, useEffect } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  Image,
  ActivityIndicator,
} from "react-native";
import Constants from "expo-constants";
import { Feather } from "@expo/vector-icons";
import { useNavigation } from "expo-router";
import { styles, Colors } from "@/assets/styles/App.styles";
import Topbar from "@/app/components/Topbar";
import Toolbar from "@/app/components/Toolbar";
import {
  useWeeklyPoints,
  useMonthlyPoints,
} from "@/app/hooks/queries/usePointsLeaderboard";
import { useUserProfileById } from "@/app/hooks/queries/useUserProfileById";
import { getImageUrl } from "@/app/utils/imageUtils";
import useAuthStore from "@/app/store/authStore";
import { useRequestFriendship } from "@/app/hooks/mutations/useFriendshipActions";
import { useFriendships } from "@/app/hooks/queries/useFriends";
import { useQueryClient } from "@tanstack/react-query";

const TournamentsScreen = () => {
  const navigation = useNavigation();
  const [activeTab, setActiveTab] = useState("weekly");
  const queryClient = useQueryClient();

  // 페이지 로딩 시 데이터 새로고침
  useEffect(() => {
    queryClient.invalidateQueries(["weeklyPoints"]);
    queryClient.invalidateQueries(["monthlyPoints"]);
    queryClient.invalidateQueries(["friendships"]);
  }, [queryClient]);

  // 쿼리 훅을 이용해 주간, 월간 리더보드 데이터를 조회합니다.
  const {
    data: weeklyData,
    isLoading: weeklyLoading,
    error: weeklyError,
    refetch: refetchWeekly,
  } = useWeeklyPoints();

  const {
    data: monthlyData,
    isLoading: monthlyLoading,
    error: monthlyError,
    refetch: refetchMonthly,
  } = useMonthlyPoints();

  // 이미 친구이거나 요청 보낸 친구들을 조회합니다.
  const { data: friendships, refetch: refetchFriendships } = useFriendships();

  // 모든 데이터를 새로고침하는 함수
  const refreshAllData = () => {
    refetchWeekly();
    refetchMonthly();
    refetchFriendships();
  };

  // 컴포넌트 마운트 시 자동으로 데이터 새로고침
  useEffect(() => {
    refreshAllData();
  }, []);

  if (weeklyLoading || monthlyLoading) {
    return (
      <View style={styles.WhiteMainContainer}>
        <Topbar title="Yarışmalar" />
        <View
          style={{ flex: 1, justifyContent: "center", alignItems: "center" }}
        >
          <ActivityIndicator size="large" color={Colors.primary} />
          <Text style={{ marginTop: 10, color: Colors.tertiary }}>
            Veriler yükleniyor...
          </Text>
        </View>
      </View>
    );
  }

  if (weeklyError || monthlyError) {
    return (
      <View style={styles.WhiteMainContainer}>
        <Topbar title="Yarışmalar" />
        <View
          style={{ flex: 1, justifyContent: "center", alignItems: "center" }}
        >
          <Text style={{ color: Colors.danger, marginBottom: 20 }}>
            Error loading leaderboard data
          </Text>
          <TouchableOpacity
            style={{
              padding: 10,
              backgroundColor: Colors.primary,
              borderRadius: 5,
            }}
            onPress={refreshAllData}
          >
            <Text style={{ color: Colors.white }}>Tekrar Dene</Text>
          </TouchableOpacity>
        </View>
      </View>
    );
  }

  // total_points 기준 내림차순 정렬 (포인트가 높은 사용자 순)
  const sortedWeeklyLeaderboard = [...(weeklyData || [])].sort(
    (a, b) => b.total_points - a.total_points
  );
  const sortedMonthlyLeaderboard = [...(monthlyData || [])].sort(
    (a, b) => b.total_points - a.total_points
  );

  // LeaderboardItem 컴포넌트
  const LeaderboardItem = ({ data, rank, friendships }) => {
    const { data: userProfile, isLoading } = useUserProfileById(data.user_id);
    const { user: authUser } = useAuthStore();
    const { mutate: sendFriendRequest } = useRequestFriendship();
    const [requestSent, setRequestSent] = useState(false);

    // 프로필 이미지 URL 해결함수
    const resolveProfileImage = () => {
      if (!userProfile) return getImageUrl("");
      const imagePath = userProfile.profile_pic || userProfile.profile_image;
      if (!imagePath) return getImageUrl("");
      return imagePath.startsWith("https") ? imagePath : getImageUrl(imagePath);
    };

    // 이미 친구거나 친구 요청이 보낸 인원 여부 확인
    // friendships API 응답 예시에는 대상 사용자의 id가 friend_id 필드에 저장됩니다.
    const isAlreadyFriend = friendships?.some(
      (friend) => friend.friend_id === userProfile?.id
    );

    // 친구 요청 버튼 클릭 시 호출
    const handleFriendRequest = () => {
      // 현재 로그인한 사용자와 대상 사용자가 다를 때 처리
      if (userProfile && authUser && userProfile.id !== authUser.id) {
        sendFriendRequest(userProfile.id, {
          onSuccess: () => {
            setRequestSent(true);
            // 친구 요청 성공 후 친구 목록 새로고침
            refetchFriendships();
          },
        });
      }
    };

    return (
      <View
        style={{
          backgroundColor: Colors.white,
          padding: 15,
          borderRadius: 10,
          marginBottom: 10,
          shadowColor: Colors.tertiary,
          shadowOffset: { width: 0, height: 2 },
          shadowOpacity: 0.1,
          shadowRadius: 4,
          elevation: 3,
        }}
      >
        {/* 상단: 사용자 정보 (순위, 프로필, 이름, 포인트) */}
        <View
          style={{
            flexDirection: "row",
            alignItems: "center",
            justifyContent: "space-between",
          }}
        >
          <View style={{ flexDirection: "row", alignItems: "center" }}>
            <Text
              style={{
                fontSize: 18,
                fontFamily: "montbold",
                color: Colors.primary,
                marginRight: 10,
              }}
            >
              {rank}
            </Text>
            {isLoading || !userProfile ? (
              <View
                style={{
                  width: 40,
                  height: 40,
                  borderRadius: 20,
                  backgroundColor: Colors.secondary,
                  marginRight: 10,
                }}
              />
            ) : (
              <Image
                source={{ uri: resolveProfileImage() }}
                style={{
                  width: 40,
                  height: 40,
                  borderRadius: 20,
                  marginRight: 10,
                }}
              />
            )}
            <Text
              style={{
                fontSize: 16,
                fontFamily: "montbold",
                color: Colors.tertiary,
              }}
            >
              {isLoading || !userProfile
                ? `User ${data.user_id}`
                : `${userProfile.first_name} ${userProfile.last_name}`}
            </Text>
          </View>
          <Text
            style={{
              fontSize: 16,
              fontFamily: "montbold",
              color: Colors.primary,
            }}
          >
            {data.total_points} Puan
          </Text>
        </View>

        {/* 하단: 친구 요청 버튼 (또는 요청 완료 메시지) – 포인트 하단에 위치 */}
        {!isLoading &&
          userProfile &&
          authUser &&
          userProfile.id !== authUser.id &&
          !requestSent &&
          !isAlreadyFriend && (
            <View style={{ alignSelf: "flex-end", marginTop: 10 }}>
              <TouchableOpacity
                onPress={handleFriendRequest}
                style={{
                  flexDirection: "row",
                  alignItems: "center",
                  backgroundColor: Colors.primary + "20",
                  paddingVertical: 5,
                  paddingHorizontal: 10,
                  borderRadius: 15,
                }}
              >
                <Feather name="user-plus" size={14} color={Colors.primary} />
                <Text
                  style={{
                    fontSize: 12,
                    fontFamily: "montmedium",
                    color: Colors.primary,
                    marginLeft: 5,
                  }}
                >
                  Arkadaş Ekle
                </Text>
              </TouchableOpacity>
            </View>
          )}
        {requestSent && (
          <View style={{ alignSelf: "flex-end", marginTop: 10 }}>
            <Text
              style={{
                fontSize: 12,
                fontFamily: "montmedium",
                color: Colors.primary,
              }}
            >
              İstek gönderildi
            </Text>
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.WhiteMainContainer}>
      <Topbar title="Yarışmalar" />
      <View style={{ flexDirection: "row", padding: 15 }}>
        <TouchableOpacity
          style={{
            flex: 1,
            padding: 10,
            alignItems: "center",
            borderBottomWidth: 2,
            borderBottomColor:
              activeTab === "weekly" ? Colors.primary : Colors.secondary,
          }}
          onPress={() => setActiveTab("weekly")}
        >
          <Text
            style={{
              fontFamily: "montbold",
              color: activeTab === "weekly" ? Colors.primary : Colors.tertiary,
            }}
          >
            Haftalık Yarışma
          </Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={{
            flex: 1,
            padding: 10,
            alignItems: "center",
            borderBottomWidth: 2,
            borderBottomColor:
              activeTab === "monthly" ? Colors.primary : Colors.secondary,
          }}
          onPress={() => setActiveTab("monthly")}
        >
          <Text
            style={{
              fontFamily: "montbold",
              color: activeTab === "monthly" ? Colors.primary : Colors.tertiary,
            }}
          >
            Aylık Yarışma
          </Text>
        </TouchableOpacity>
      </View>

      <ScrollView
        style={{ flex: 1, padding: 15 }}
        onRefresh={refreshAllData}
        refreshing={weeklyLoading || monthlyLoading}
      >
        {activeTab === "weekly"
          ? sortedWeeklyLeaderboard.map((item, index) => (
              <LeaderboardItem
                key={item.user_id}
                data={item}
                rank={index + 1}
                friendships={friendships}
              />
            ))
          : sortedMonthlyLeaderboard.map((item, index) => (
              <LeaderboardItem
                key={item.user_id}
                data={item}
                rank={index + 1}
                friendships={friendships}
              />
            ))}
      </ScrollView>
      <Toolbar />
    </View>
  );
};

export default TournamentsScreen;
