import {
  Text,
  View,
  TouchableOpacity,
  SafeAreaView,
  ScrollView,
  StyleSheet,
} from "react-native";
import { useState } from "react";
import { TextInput } from "react-native-paper";
import { useRouter } from "expo-router";
import Toast from "@/app/components/common/ToastMessage";
import { useForgotPassword } from "@/app/hooks/mutations/useForgotPassword";

export default function ForgotPasswordScreen() {
  const [email, setEmail] = useState("");
  const router = useRouter();
  const { mutate: forgotPassword, isLoading } = useForgotPassword();

  const handleSubmit = () => {
    if (!email || !email.includes("@")) {
      Toast("Hata", "Geçerli bir e-posta adresi giriniz.", "danger");
      return;
    }

    forgotPassword(email, {
      onSuccess: (data) => {
        // If the response message is "Reset code has been sent to your email address."
        // display a Turkish message.
        Toast(
          "Başarılı",
          "E-posta adresinize sıfırlama kodu gönderildi.",
          "success"
        );
        router.push({
          pathname: "pages/login/forgotPasswordCode",
          params: { email },
        });
      },
      onError: (error) => {
        const detail = error.response?.data?.detail;
        if (detail) {
          if (detail.includes("User with provided email not found")) {
            Toast(
              "Hata",
              "Girilen e-posta adresine ait kullanıcı bulunamadı.",
              "danger"
            );
          } else {
            Toast("Hata", detail, "danger");
          }
        } else {
          Toast("Hata", error.message || "Bir hata oluştu.", "danger");
        }
      },
    });
  };

  return (
    <SafeAreaView style={styles.MainContainer}>
      <ScrollView
        contentContainerStyle={{ flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={false}
      >
        <View style={[styles.RegisterContainer, { justifyContent: "center" }]}>
          <Text style={styles.RegisterTitle}>Şifremi Unuttum</Text>

          <Text style={styles.EmailVerificationText}>
            Lütfen hesabınıza ait e-posta adresini giriniz. Şifre sıfırlama kodu
            e-posta adresinize gönderilecektir.
          </Text>

          <View style={styles.RegisterInputContainer}>
            <TextInput
              mode="outlined"
              style={styles.RegisterInput}
              label="E-posta"
              value={email}
              onChangeText={setEmail}
              keyboardType="email-address"
              autoCapitalize="none"
              activeOutlineColor={"#7AB52E"}
              outlineColor={"#2E4D00"}
              left={<TextInput.Icon icon="email" color={"#7AB52E"} />}
            />
          </View>

          <TouchableOpacity
            style={styles.RegisterButton}
            onPress={handleSubmit}
            activeOpacity={0.6}
          >
            <Text style={styles.RegisterButtonText}>Kod Gönder</Text>
          </TouchableOpacity>

          <TouchableOpacity
            style={styles.RegisterBackButton}
            onPress={() => router.back()}
          >
            <Text style={styles.RegisterBackButtonText}>Geri Dön</Text>
          </TouchableOpacity>
        </View>
      </ScrollView>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  MainContainer: {
    flex: 1,
    width: "100%",
  },
  RegisterContainer: {
    flex: 1,
    padding: 20,
    justifyContent: "center",
  },
  RegisterTitle: {
    fontSize: 24,
    fontFamily: "montbold",
    color: "#2D2D2D",
    marginBottom: 30,
    textAlign: "center",
  },
  EmailVerificationText: {
    fontSize: 14,
    color: "#2D2D2D",
    textAlign: "center",
    marginBottom: 20,
    fontFamily: "montmedium",
    paddingHorizontal: 20,
  },
  RegisterInputContainer: {
    gap: 15,
  },
  RegisterInput: {
    backgroundColor: "#fff",
    fontSize: 16,
    fontFamily: "montregular",
  },
  RegisterButton: {
    backgroundColor: "#7AB52E",
    padding: 15,
    borderRadius: 10,
    alignItems: "center",
    marginTop: 30,
  },
  RegisterButtonText: {
    color: "#fff",
    fontSize: 16,
    fontFamily: "montbold",
  },
  RegisterBackButton: {
    padding: 15,
    borderRadius: 10,
    alignItems: "center",
    marginTop: 10,
  },
  RegisterBackButtonText: {
    color: "#7AB52E",
    fontSize: 16,
    fontFamily: "montmedium",
  },
});
