import {
  Text,
  View,
  TouchableOpacity,
  SafeAreaView,
  ScrollView,
  StyleSheet,
} from "react-native";
import { useState } from "react";
import { TextInput } from "react-native-paper";
import { useRouter, useLocalSearchParams } from "expo-router";
import Toast from "@/app/components/common/ToastMessage";
import { useVerifyResetCode } from "@/app/hooks/mutations/useVerifyResetCode";

export default function ForgotPasswordCodeScreen() {
  const [code, setCode] = useState("");
  const router = useRouter();
  const { email } = useLocalSearchParams(); // useLocalSearchParams를 사용합니다.
  const { mutate: verifyResetCode } = useVerifyResetCode();

  const handleSubmit = () => {
    if (code.length !== 6) {
      Toast("Hata", "Lütfen 6 haneli kodu giriniz.", "danger");
      return;
    }

    if (!email) {
      Toast("Hata", "E-posta bilgisi eksik.", "danger");
      return;
    }

    verifyResetCode(
      { email, reset_code: code },
      {
        onSuccess: () => {
          Toast("Başarılı", "Kod doğrulandı.");
          router.push({
            pathname: "pages/login/forgotPasswordNewPassword",
            params: { email, reset_code: code },
          });
        },
        onError: (error) => {
          const detail = error.response?.data?.detail;
          Toast("Hata", detail || "Kod doğrulanamadı.", "danger");
        },
      }
    );
  };

  return (
    <SafeAreaView style={styles.MainContainer}>
      <ScrollView
        contentContainerStyle={{ flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={false}
      >
        <View style={[styles.RegisterContainer, { justifyContent: "center" }]}>
          <Text style={styles.RegisterTitle}>Kod Doğrulama</Text>

          <Text style={styles.EmailVerificationText}>
            E-posta adresinize gönderilen 6 haneli doğrulama kodunu giriniz.
          </Text>

          <View style={styles.RegisterInputContainer}>
            <TextInput
              mode="outlined"
              style={styles.RegisterInput}
              label="Doğrulama Kodu"
              value={code}
              onChangeText={setCode}
              keyboardType="numeric"
              maxLength={6}
              activeOutlineColor={"#7AB52E"}
              outlineColor={"#2E4D00"}
              left={<TextInput.Icon icon="numeric" color={"#7AB52E"} />}
            />
          </View>

          <TouchableOpacity
            style={styles.RegisterButton}
            onPress={handleSubmit}
            activeOpacity={0.6}
          >
            <Text style={styles.RegisterButtonText}>Doğrula</Text>
          </TouchableOpacity>

          <TouchableOpacity
            style={styles.RegisterBackButton}
            onPress={() => router.back()}
          >
            <Text style={styles.RegisterBackButtonText}>Geri Dön</Text>
          </TouchableOpacity>
        </View>
      </ScrollView>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  MainContainer: {
    flex: 1,
    width: "100%",
  },
  RegisterContainer: {
    flex: 1,
    padding: 20,
    justifyContent: "center",
  },
  RegisterTitle: {
    fontSize: 24,
    fontFamily: "montbold",
    color: "#2D2D2D",
    marginBottom: 30,
    textAlign: "center",
  },
  EmailVerificationText: {
    fontSize: 14,
    color: "#2D2D2D",
    textAlign: "center",
    marginBottom: 20,
    fontFamily: "montmedium",
    paddingHorizontal: 20,
  },
  RegisterInputContainer: {
    gap: 15,
  },
  RegisterInput: {
    backgroundColor: "#fff",
    fontSize: 16,
    fontFamily: "montregular",
  },
  RegisterButton: {
    backgroundColor: "#7AB52E",
    padding: 15,
    borderRadius: 10,
    alignItems: "center",
    marginTop: 30,
  },
  RegisterButtonText: {
    color: "#fff",
    fontSize: 16,
    fontFamily: "montbold",
  },
  RegisterBackButton: {
    padding: 15,
    borderRadius: 10,
    alignItems: "center",
    marginTop: 10,
  },
  RegisterBackButtonText: {
    color: "#7AB52E",
    fontSize: 16,
    fontFamily: "montmedium",
  },
});
