import React, { useState } from "react";
import {
  Text,
  View,
  TouchableOpacity,
  SafeAreaView,
  ScrollView,
  StyleSheet,
} from "react-native";
import { TextInput } from "react-native-paper";
import { useRouter, useLocalSearchParams } from "expo-router";
import Toast from "@/app/components/common/ToastMessage";
import { useChangePassword } from "@/app/hooks/mutations/useChangePassword";

export default function ForgotPasswordNewPasswordScreen() {
  const [formData, setFormData] = useState({
    password: "",
    confirmPassword: "",
  });
  const router = useRouter();

  // email ve reset_code, bir önceki sayfadan URL parametresi olarak gönderiliyor.
  const { email, reset_code } = useLocalSearchParams();
  const { mutate: changePassword } = useChangePassword();

  const handleSubmit = () => {
    if (formData.password.length < 6) {
      Toast("Hata", "Şifre en az 6 karakter olmalıdır.", "danger");
      return;
    }

    if (formData.password !== formData.confirmPassword) {
      Toast("Hata", "Şifreler eşleşmiyor.", "danger");
      return;
    }

    if (!email || !reset_code) {
      Toast("Hata", "Gerekli bilgiler eksik.", "danger");
      return;
    }

    // useChangePassword hook'u ile backend'e POST isteği gönderiyoruz.
    changePassword(
      {
        email,
        reset_code,
        new_password: formData.password,
        confirm_password: formData.confirmPassword,
      },
      {
        onSuccess: () => {
          Toast("Başarılı", "Şifreniz başarıyla değiştirildi.", "success");
          router.replace("pages/login/loginScreen");
        },
        onError: (error) => {
          const detail = error.response?.data?.detail;
          Toast("Hata", detail || "Şifre değiştirilemedi.", "danger");
        },
      }
    );
  };

  return (
    <SafeAreaView style={styles.MainContainer}>
      <ScrollView
        contentContainerStyle={{ flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={false}
      >
        <View style={[styles.RegisterContainer, { justifyContent: "center" }]}>
          <Text style={styles.RegisterTitle}>Yeni Şifre</Text>

          <Text style={styles.EmailVerificationText}>
            Lütfen yeni şifrenizi belirleyiniz.
          </Text>

          <View style={styles.RegisterInputContainer}>
            <TextInput
              mode="outlined"
              style={styles.RegisterInput}
              label="Yeni Şifre"
              value={formData.password}
              onChangeText={(text) =>
                setFormData({ ...formData, password: text })
              }
              secureTextEntry
              activeOutlineColor={"#7AB52E"}
              outlineColor={"#2E4D00"}
              left={<TextInput.Icon icon="lock" color={"#7AB52E"} />}
            />

            <TextInput
              mode="outlined"
              style={styles.RegisterInput}
              label="Şifre Tekrar"
              value={formData.confirmPassword}
              onChangeText={(text) =>
                setFormData({ ...formData, confirmPassword: text })
              }
              secureTextEntry
              activeOutlineColor={"#7AB52E"}
              outlineColor={"#2E4D00"}
              left={<TextInput.Icon icon="lock" color={"#7AB52E"} />}
            />
          </View>

          <TouchableOpacity
            style={styles.RegisterButton}
            onPress={handleSubmit}
            activeOpacity={0.6}
          >
            <Text style={styles.RegisterButtonText}>Şifreyi Değiştir</Text>
          </TouchableOpacity>

          <TouchableOpacity
            style={styles.RegisterBackButton}
            onPress={() => router.back()}
          >
            <Text style={styles.RegisterBackButtonText}>Geri Dön</Text>
          </TouchableOpacity>
        </View>
      </ScrollView>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  MainContainer: {
    flex: 1,
    width: "100%",
  },
  RegisterContainer: {
    flex: 1,
    padding: 20,
    justifyContent: "center",
  },
  RegisterTitle: {
    fontSize: 24,
    fontFamily: "montbold",
    color: "#2D2D2D",
    marginBottom: 30,
    textAlign: "center",
  },
  EmailVerificationText: {
    fontSize: 14,
    color: "#2D2D2D",
    textAlign: "center",
    marginBottom: 20,
    fontFamily: "montmedium",
    paddingHorizontal: 20,
  },
  RegisterInputContainer: {
    gap: 15,
  },
  RegisterInput: {
    backgroundColor: "#fff",
    fontSize: 16,
    fontFamily: "montregular",
  },
  RegisterButton: {
    backgroundColor: "#7AB52E",
    padding: 15,
    borderRadius: 10,
    alignItems: "center",
    marginTop: 30,
  },
  RegisterButtonText: {
    color: "#fff",
    fontSize: 16,
    fontFamily: "montbold",
  },
  RegisterBackButton: {
    padding: 15,
    borderRadius: 10,
    alignItems: "center",
    marginTop: 10,
  },
  RegisterBackButtonText: {
    color: "#7AB52E",
    fontSize: 16,
    fontFamily: "montmedium",
  },
});
