import React, { useState, useEffect } from "react";
import {
  Text,
  View,
  TouchableOpacity,
  SafeAreaView,
  ScrollView,
  Image,
  FlatList,
  KeyboardAvoidingView,
  Platform,
} from "react-native";
import { TextInput, RadioButton } from "react-native-paper";
import { useNavigation } from "expo-router";
import { FontAwesome5 } from "@expo/vector-icons";
import * as ImagePicker from "expo-image-picker";
import Toast from "@/app/components/common/ToastMessage";
import { Colors, styles } from "@/assets/styles/App.styles";
import api from "@/app/lib/axios";
import PhoneInput, {
  getCountryByCca2,
} from "react-native-international-phone-number";
import { registerUser } from "@/app/api/endpoints/users";
import { useReferralCodes } from "@/app/hooks/queries/useReferralCodes";
import DateTimePicker from "@react-native-community/datetimepicker";

export default function RegisterScreen() {
  const [formData, setFormData] = useState({
    name: "",
    surname: "",
    phone: "",
    email: "",
    password: "",
    gender: "1",
    age: "",
    country: "",
    image: null,
    referanceCode: "",
  });

  const [countryCode, setCountryCode] = useState("TR");
  const [countryName, setCountryName] = useState("");
  const [showDatePicker, setShowDatePicker] = useState(false);

  // Update handleDateChange function
  const handleDateChange = (event, selectedDate) => {
    console.log(
      "Date change event:",
      event?.type || "manual",
      "Selected date:",
      selectedDate
    );

    // Only hide date picker on Android when canceled
    if (Platform.OS === "android" && event.type === "dismissed") {
      setShowDatePicker(false);
      return;
    }

    if (selectedDate) {
      const year = selectedDate.getFullYear();
      const month = String(selectedDate.getMonth() + 1).padStart(2, "0");
      const day = String(selectedDate.getDate()).padStart(2, "0");
      const formattedDate = `${year}-${month}-${day}`;
      console.log("Formatted date:", formattedDate);
      setFormData({ ...formData, age: formattedDate });
    }
  };

  const pickImage = async () => {
    const result = await ImagePicker.launchImageLibraryAsync({
      mediaTypes: ImagePicker.MediaTypeOptions.Images,
      allowsEditing: true,
      aspect: [1, 1],
      quality: 1,
    });

    if (!result.canceled) {
      setFormData({ ...formData, image: result.assets[0].uri });
    }
  };

  const navigation = useNavigation();
  const { data: referralCodes, isLoading: loadingReferral } =
    useReferralCodes();

  const handleRegister = async () => {
    console.log("Register button pressed");

    if (
      !formData.name ||
      !formData.surname ||
      !formData.email ||
      !formData.password ||
      !formData.phone ||
      !formData.country ||
      !formData.age
    ) {
      Toast("Error", "Please fill all required fields.", "danger");
      return;
    }

    if (!validateBirthDate(formData.age)) {
      Toast(
        "Error",
        "Lütfen geçerli bir doğum tarihi giriniz (YYYY-MM-DD).",
        "danger"
      );
      return;
    }

    if (formData.referanceCode) {
      if (loadingReferral) {
        Toast(
          "Hata",
          "Referans kodları yükleniyor, lütfen sonra tekrar deneyin.",
          "danger"
        );
        return;
      }
      if (!referralCodes) {
        Toast(
          "Hata",
          "Referans kodları alınamadı, lütfen daha sonra tekrar deneyin.",
          "danger"
        );
        return;
      }
      const isValidReferral = referralCodes.some(
        (code) => code.reference_code === formData.referanceCode
      );
      if (!isValidReferral) {
        Toast("Hata", "Geçersiz referans kodu.", "danger");
        return;
      }
    }

    const payloadData = {
      email: formData.email,
      first_name: formData.name,
      last_name: formData.surname,
      phone_code: "+90",
      phone: formData.phone,
      password: formData.password,
      country_code:
        typeof formData.country === "object"
          ? formData.country.cca2
          : formData.country,
      gender: formData.gender === "1" ? "M" : "F",
      birth_date: formData.age,
      reference_code_used: formData.referanceCode || null,
    };

    if (formData.image && formData.image.startsWith("file://")) {
      payloadData.profileImage = {
        uri: formData.image,
        name: "profile_image.jpg",
        type: "image/jpeg",
      };
    }

    try {
      const response = await registerUser(payloadData);
      console.log("Registration response:", response);
      Toast("Success", "Kayıt işlemi başarıyla tamamlandı.");
      navigation.navigate("pages/login/loginScreen");
    } catch (error) {
      console.error("Registration error:", error);
      Toast("Error", error.message || "Registration failed.", "danger");
    }
  };

  const handleCountryChange = (selectedCountry) => {
    let isoCode = selectedCountry;
    if (
      selectedCountry &&
      typeof selectedCountry === "object" &&
      selectedCountry.cca2
    ) {
      isoCode = selectedCountry.cca2;
    }
    setCountryCode(isoCode);
    setFormData({ ...formData, country: isoCode });
  };

  useEffect(() => {
    const countryObj = getCountryByCca2(countryCode);
    if (countryObj) {
      setCountryName(countryObj.callingCode);
    }
  }, [countryCode]);

  const validateBirthDate = (dateStr) => {
    if (!/^\d{4}-\d{2}-\d{2}$/.test(dateStr)) return false;
    const [year, month, day] = dateStr
      .split("-")
      .map((num) => parseInt(num, 10));
    const date = new Date(year, month - 1, day);
    if (
      date.getFullYear() !== year ||
      date.getMonth() !== month - 1 ||
      date.getDate() !== day
    )
      return false;
    return true;
  };

  const handleBirthDateChange = (text) => {
    const digits = text.replace(/[^0-9]/g, "");
    if (digits.length === 8) {
      const formatted = `${digits.slice(0, 4)}-${digits.slice(
        4,
        6
      )}-${digits.slice(6, 8)}`;
      setFormData({ ...formData, age: formatted });
    } else {
      setFormData({ ...formData, age: text });
    }
  };

  return (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === "ios" ? "padding" : "height"}
      keyboardVerticalOffset={Platform.OS === "ios" ? 64 : 0}
    >
      <SafeAreaView style={styles.MainContainer}>
        <ScrollView
          contentContainerStyle={{ flexGrow: 1 }}
          keyboardShouldPersistTaps="handled"
          showsVerticalScrollIndicator={false}
        >
          <View style={styles.RegisterContainer}>
            <Text style={styles.RegisterTitle}>Kullanıcı Kayıt</Text>

            <View style={styles.RegisterInputContainer}>
              <View
                style={{
                  width: "100%",
                  borderRadius: 10,
                  justifyContent: "center",
                  alignItems: "center",
                }}
              >
                <TouchableOpacity
                  style={{
                    width: 125,
                    height: 125,
                    borderRadius: 100,
                    backgroundColor: Colors.lightGray,
                    justifyContent: "center",
                    alignItems: "center",
                  }}
                  onPress={pickImage}
                >
                  {formData.image ? (
                    <Image
                      source={{ uri: formData.image }}
                      style={{
                        width: "100%",
                        height: "100%",
                        borderRadius: 100,
                      }}
                    />
                  ) : (
                    <FontAwesome5
                      name="camera"
                      size={24}
                      color={Colors.primary}
                    />
                  )}
                </TouchableOpacity>
              </View>
              <TextInput
                mode="outlined"
                style={styles.RegisterInput}
                label="Ad"
                value={formData.name}
                onChangeText={(text) =>
                  setFormData({ ...formData, name: text })
                }
                activeOutlineColor={Colors.primary}
                outlineColor={Colors.secondary}
                left={<TextInput.Icon icon="account" color={Colors.primary} />}
              />
              <TextInput
                mode="outlined"
                style={styles.RegisterInput}
                label="Soyad"
                value={formData.surname}
                onChangeText={(text) =>
                  setFormData({ ...formData, surname: text })
                }
                activeOutlineColor={Colors.primary}
                outlineColor={Colors.secondary}
                left={<TextInput.Icon icon="account" color={Colors.primary} />}
              />
              <View
                style={{
                  flexDirection: "row",
                  width: "100%",
                  alignItems: "center",
                  marginVertical: 8,
                }}
              >
                <View
                  style={{
                    width: "37%",
                    marginRight: 5,
                    borderWidth: 1,
                    borderColor: Colors.secondary,
                    borderRadius: 10,
                    overflow: "hidden",
                    height: 50,
                    backgroundColor: "white", // 배경색 추가
                  }}
                >
                  <PhoneInput
                    value={countryName}
                    onChangePhoneNumber={() => {}}
                    selectedCountry={getCountryByCca2(countryCode)}
                    onChangeSelectedCountry={handleCountryChange}
                    defaultCountry="TR"
                    language="tr"
                    editable={false}
                    defaultValue={countryName}
                    containerStyle={{
                      width: "100%",
                      height: "100%",
                      backgroundColor: "transparent",
                      alignItems: "center",
                      justifyContent: "center",
                      flexDirection: "row",
                      position: "relative",
                      zIndex: 1,
                    }}
                    flagContainerStyle={{
                      backgroundColor: "transparent",
                      justifyContent: "center",
                      alignItems: "center",
                      marginLeft: 8,
                      height: "100%",
                      position: "relative",
                    }}
                    textContainerStyle={{
                      backgroundColor: "transparent",
                      paddingLeft: 8,
                      paddingRight: 8,
                      justifyContent: "center",
                      height: "100%",
                      position: "relative",
                    }}
                    textInputStyle={{
                      padding: 0,
                      margin: 0,
                      height: "100%",
                      color: Colors.tertiary,
                      fontSize: 14,
                      fontFamily: "montmedium",
                    }}
                  />
                </View>

                <View style={{ width: "62%", marginRight: 10 }}>
                  <TextInput
                    mode="outlined"
                    style={[styles.RegisterInput, { marginTop: -5 }]}
                    label="Telefon"
                    value={formData.phone}
                    onChangeText={(text) =>
                      setFormData({ ...formData, phone: text })
                    }
                    keyboardType="number-pad"
                    activeOutlineColor={Colors.primary}
                    outlineColor={Colors.secondary}
                    left={
                      <TextInput.Icon icon="phone" color={Colors.primary} />
                    }
                  />
                </View>
              </View>
              <TextInput
                mode="outlined"
                style={styles.RegisterInput}
                label="E-posta"
                value={formData.email}
                onChangeText={(text) =>
                  setFormData({ ...formData, email: text })
                }
                keyboardType="email-address"
                autoCapitalize="none"
                activeOutlineColor={Colors.primary}
                outlineColor={Colors.secondary}
                left={<TextInput.Icon icon="email" color={Colors.primary} />}
              />
              <TextInput
                mode="outlined"
                style={styles.RegisterInput}
                label="Şifre"
                value={formData.password}
                onChangeText={(text) =>
                  setFormData({ ...formData, password: text })
                }
                keyboardType="default"
                secureTextEntry
                autoCapitalize="none"
                activeOutlineColor={Colors.primary}
                outlineColor={Colors.secondary}
                left={<TextInput.Icon icon="lock" color={Colors.primary} />}
              />
              <View style={styles.RegisterGenderContainer}>
                <Text style={styles.RegisterLabel}>Cinsiyet</Text>
                <View style={styles.RegisterGenderOptions}>
                  <View style={styles.RegisterGenderOption}>
                    <RadioButton
                      value="1"
                      status={formData.gender === "1" ? "checked" : "unchecked"}
                      onPress={() => setFormData({ ...formData, gender: "1" })}
                      color={Colors.primary}
                    />
                    <Text style={{ fontSize: 12 }}>Erkek</Text>
                  </View>
                  <View style={styles.RegisterGenderOption}>
                    <RadioButton
                      value="2"
                      status={formData.gender === "2" ? "checked" : "unchecked"}
                      onPress={() => setFormData({ ...formData, gender: "2" })}
                      color={Colors.primary}
                    />
                    <Text style={{ fontSize: 12 }}>Kadın</Text>
                  </View>
                </View>
              </View>
              <TouchableOpacity
                onPress={() => {
                  console.log("Date picker button pressed");
                  setShowDatePicker(true);
                }}
                style={{ width: "100%" }}
              >
                <TextInput
                  mode="outlined"
                  style={[styles.RegisterInput, { color: Colors.black }]}
                  label="Doğum Tarihi"
                  value={formData.age}
                  editable={false}
                  activeOutlineColor={Colors.primary}
                  outlineColor={Colors.secondary}
                  left={
                    <TextInput.Icon
                      icon="calendar"
                      color={Colors.primary}
                      onPress={() => {
                        console.log("Calendar icon pressed");
                        setShowDatePicker(true);
                      }}
                    />
                  }
                  textColor={Colors.black}
                />
              </TouchableOpacity>

              {/* DateTimePicker component */}
              {showDatePicker &&
                (Platform.OS === "ios" ? (
                  <View
                    style={{
                      width: "100%",
                      backgroundColor: "#f5f5f5",
                      padding: 10,
                      borderRadius: 10,
                      zIndex: 1000,
                      elevation: 5,
                      shadowColor: "#000",
                      shadowOffset: { width: 0, height: 2 },
                      shadowOpacity: 0.25,
                      shadowRadius: 3.84,
                      marginVertical: 10,
                    }}
                  >
                    <Text
                      style={{
                        fontSize: 16,
                        fontWeight: "bold",
                        textAlign: "center",
                        marginBottom: 10,
                        color: Colors.primary,
                      }}
                    >
                      Doğum Tarihi Seçin
                    </Text>

                    {/* iOS 날짜 선택기에서 spinner 모드로 변경하고 색상 조정 */}
                    <DateTimePicker
                      testID="dateTimePicker"
                      value={
                        formData.age && validateBirthDate(formData.age)
                          ? new Date(formData.age)
                          : new Date(2000, 0, 1)
                      }
                      mode="date"
                      display="spinner"
                      onChange={(event, date) => {
                        console.log("iOS date change:", event, date);
                        handleDateChange(event, date);
                      }}
                      maximumDate={new Date()}
                      minimumDate={new Date(1900, 0, 1)}
                      textColor="#000000"
                      themeVariant="light"
                      accentColor={Colors.primary}
                      style={{
                        height: 180,
                        width: "100%",
                      }}
                    />

                    {/* 추가 UI: 선택한 날짜 미리보기 */}
                    {formData.age && (
                      <View
                        style={{
                          backgroundColor: "white",
                          padding: 10,
                          borderRadius: 8,
                          marginVertical: 8,
                          alignItems: "center",
                        }}
                      >
                        <Text
                          style={{
                            fontSize: 16,
                            fontWeight: "600",
                            color: Colors.primary,
                          }}
                        >
                          {formData.age.split("-").reverse().join("/")}
                        </Text>
                      </View>
                    )}

                    <View
                      style={{
                        flexDirection: "row",
                        justifyContent: "space-between",
                        marginTop: 15,
                      }}
                    >
                      <TouchableOpacity
                        onPress={() => {
                          console.log("Cancel button pressed");
                          setShowDatePicker(false);
                        }}
                        style={{
                          padding: 12,
                          backgroundColor: "#f8f8f8",
                          borderRadius: 6,
                          width: "48%",
                          alignItems: "center",
                        }}
                      >
                        <Text
                          style={{
                            color: Colors.danger,
                            fontSize: 16,
                            fontWeight: "600",
                          }}
                        >
                          İptal
                        </Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        onPress={() => {
                          console.log("Confirm button pressed");
                          setShowDatePicker(false);
                        }}
                        style={{
                          padding: 12,
                          backgroundColor: Colors.primary,
                          borderRadius: 6,
                          width: "48%",
                          alignItems: "center",
                        }}
                      >
                        <Text
                          style={{
                            color: "white",
                            fontSize: 16,
                            fontWeight: "600",
                          }}
                        >
                          Tamam
                        </Text>
                      </TouchableOpacity>
                    </View>
                  </View>
                ) : (
                  <DateTimePicker
                    testID="dateTimePicker"
                    value={
                      formData.age && validateBirthDate(formData.age)
                        ? new Date(formData.age)
                        : new Date(2000, 0, 1)
                    }
                    mode="date"
                    display="default"
                    onChange={handleDateChange}
                    maximumDate={new Date()}
                    minimumDate={new Date(1900, 0, 1)}
                  />
                ))}
              <Text
                style={{
                  fontSize: 14,
                  color: Colors.secondary,
                  marginTop: 10,
                  marginBottom: -10,
                }}
              >
                Varsa Referans Kodunu Giriniz
              </Text>
              <TextInput
                mode="outlined"
                style={[styles.RegisterInput, { marginTop: 0 }]}
                label="Referans Kodu"
                value={formData.referanceCode}
                onChangeText={(text) =>
                  setFormData({ ...formData, referanceCode: text })
                }
                keyboardType="default"
                activeOutlineColor={Colors.primary}
                outlineColor={Colors.secondary}
                left={<TextInput.Icon icon="link" color={Colors.primary} />}
              />
            </View>

            <TouchableOpacity
              style={styles.RegisterButton}
              onPress={handleRegister}
              activeOpacity={0.6}
            >
              <Text style={styles.RegisterButtonText}>Kayıt Ol</Text>
            </TouchableOpacity>

            <TouchableOpacity
              style={styles.RegisterBackButton}
              onPress={() => navigation.goBack()}
            >
              <Text style={styles.RegisterBackButtonText}>Geri Dön</Text>
            </TouchableOpacity>
          </View>
        </ScrollView>
      </SafeAreaView>
    </KeyboardAvoidingView>
  );
}
