import React, { useState, useEffect } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  Image,
  SafeAreaView,
  ActivityIndicator,
} from "react-native";
import { TextInput } from "react-native-paper";
import { Feather } from "@expo/vector-icons";
import { useNavigation } from "expo-router";
import { styles, Colors } from "@/assets/styles/App.styles";
import Topbar from "@/app/components/Topbar";
import Toolbar from "@/app/components/Toolbar";
import Toast from "@/app/components/common/ToastMessage";
import * as ImagePicker from "expo-image-picker";

import {
  useUserProfile,
  useUpdateProfile,
} from "@/app/hooks/queries/useProfile";
import api from "@/app/lib/axios";
import useAuthStore from "@/app/store/authStore";

function EditProfileScreen() {
  const navigation = useNavigation();

  // Initialize form state.
  // We're using the keys that match our API payload:
  // full_name, email, phone, daily_step_goal and profile_image.
  const [formData, setFormData] = useState({
    full_name: "",
    email: "",
    phone: "",
    daily_step_goal: "",
    profile_image: "", // Holds the image URL (if not updated) or local URI (if updated)
  });

  const { data: profile, isLoading, error, refetch } = useUserProfile();
  const { mutate: updateProfile } = useUpdateProfile();

  // When profile data becomes available, update the local state.
  useEffect(() => {
    if (profile) {
      setFormData({
        // Combining first_name and last_name as full_name.
        full_name: `${profile.first_name} ${profile.last_name}`,
        email: profile.email,
        phone: profile.phone,
        daily_step_goal: profile.daily_step_goal.toString(),
        // Use the computed URL (profile_pic) to display the image.
        profile_image: profile.profile_pic || "https://i.pravatar.cc/200",
      });
    } else {
      refetch();
    }
  }, [profile]);

  // Launch the image picker and update the profile_image key with the local URI
  const pickImage = async () => {
    let result = await ImagePicker.launchImageLibraryAsync({
      mediaTypes: ImagePicker.MediaTypeOptions.Images,
      // mediaTypes: ImagePicker.MediaType.Images,
      allowsEditing: true,
    });
    if (!result.canceled) {
      // Save the local URI in profile_image.
      setFormData({ ...formData, profile_image: result.assets[0].uri });
    }
  };

  // Handle the save action using a FormData payload.
  const handleSave = async () => {
    // 필수 필드 확인
    if (
      !formData.full_name ||
      !formData.email ||
      !formData.phone ||
      !formData.daily_step_goal
    ) {
      Toast("Hata", "Lütfen tüm zorunlu alanları doldurunuz.", "danger");
      return;
    }

    // full_name 검증 (예시)
    const fullNameRegex = /^[a-zA-Z]+\s+[a-zA-Z]+$/;
    if (!fullNameRegex.test(formData.full_name)) {
      Toast(
        "Hata",
        "İsim ve soyisim yalnızca alfabetik karakterlerden oluşmalıdır.",
        "danger"
      );
      return;
    }

    // Retrieve the authenticated user ID from the auth store.
    const { user: authUser } = useAuthStore.getState();
    if (!authUser || !authUser.id) {
      Toast("Hata", "Kullanıcı bilgileri alınamadı.", "danger");
      return;
    }

    try {
      const userData = {
        full_name: formData.full_name,
        email: formData.email,
        phone: formData.phone,
        daily_step_goal: Number(formData.daily_step_goal),
      };

      // Always use FormData
      const payload = new FormData();
      payload.append("user_in", JSON.stringify(userData));

      // Append image only if a new image is chosen.
      if (formData.profile_image.startsWith("file://")) {
        payload.append("profile_image", {
          uri: formData.profile_image,
          name: "profile_image.jpg",
          type: "image/jpeg",
        });
      }

      updateProfile(payload, {
        onSuccess: () => {
          Toast("Başarılı", "Profiliniz başarıyla güncellendi.");
          refetch();
          navigation.goBack();
        },
      });
    } catch (err) {
      console.error("Profile update error:", err);
      if (err.response && err.response.data) {
        console.error("Validation errors:", err.response.data);
      }
    }
  };

  if (isLoading) {
    return (
      <View style={{ flex: 1, justifyContent: "center", alignItems: "center" }}>
        <ActivityIndicator size="large" color={Colors.primary} />
      </View>
    );
  }

  if (error) {
    return (
      <View style={{ flex: 1, justifyContent: "center", alignItems: "center" }}>
        <Text>Hata: {error.message || "Profil bilgileri alınamadı."}</Text>
      </View>
    );
  }

  // If profile isn't available, show an appropriate message.
  if (!profile) {
    return (
      <View style={{ flex: 1, justifyContent: "center", alignItems: "center" }}>
        <Text>Profil bilgisi bulunamadı.</Text>
      </View>
    );
  }

  return (
    <SafeAreaView style={styles.MainContainer}>
      <Topbar title="Profili Düzenle" />
      <ScrollView
        style={{ width: "100%" }}
        showsVerticalScrollIndicator={false}
        contentContainerStyle={{ padding: 20 }}
      >
        <View style={styles.profilePageHeaderContainer}>
          <TouchableOpacity activeOpacity={0.8} onPress={pickImage}>
            <Image
              source={{
                uri: formData.profile_image || "https://i.pravatar.cc/200",
              }}
              style={styles.profilePageAvatar}
            />
            <View style={styles.profilePageEditAvatarButton}>
              <Feather name="camera" size={20} color={Colors.white} />
            </View>
          </TouchableOpacity>
        </View>

        <View style={styles.RegisterInputContainer}>
          <TextInput
            mode="outlined"
            style={styles.RegisterInput}
            label="Ad Soyad"
            value={formData.full_name}
            onChangeText={(text) =>
              setFormData({ ...formData, full_name: text })
            }
            activeOutlineColor={Colors.primary}
            outlineColor={Colors.secondary}
            left={<TextInput.Icon icon="account" color={Colors.primary} />}
          />

          <TextInput
            mode="outlined"
            style={styles.RegisterInput}
            label="E-posta"
            value={formData.email}
            onChangeText={(text) => setFormData({ ...formData, email: text })}
            keyboardType="email-address"
            activeOutlineColor={Colors.primary}
            outlineColor={Colors.secondary}
            left={<TextInput.Icon icon="email" color={Colors.primary} />}
          />

          <TextInput
            mode="outlined"
            style={styles.RegisterInput}
            label="Telefon"
            value={formData.phone}
            onChangeText={(text) => setFormData({ ...formData, phone: text })}
            keyboardType="phone-pad"
            activeOutlineColor={Colors.primary}
            outlineColor={Colors.secondary}
            left={<TextInput.Icon icon="phone" color={Colors.primary} />}
          />

          <TextInput
            mode="outlined"
            style={styles.RegisterInput}
            label="Günlük Hedef"
            value={formData.daily_step_goal}
            onChangeText={(text) =>
              setFormData({ ...formData, daily_step_goal: text })
            }
            keyboardType="numeric"
            activeOutlineColor={Colors.primary}
            outlineColor={Colors.secondary}
            left={<TextInput.Icon icon="target" color={Colors.primary} />}
            right={
              <TextInput.Affix
                text="Adım"
                textStyle={{
                  fontSize: 14,
                  fontFamily: "montsemibold",
                  color: Colors.tertiary,
                }}
              />
            }
          />
        </View>

        <TouchableOpacity
          style={[
            styles.profilePageEditButton,
            { marginTop: 30, marginBottom: 10 },
          ]}
          onPress={handleSave}
          activeOpacity={0.6}
        >
          <Feather name="save" size={24} color={Colors.white} />
          <Text style={[styles.profilePageEditButtonText, { marginLeft: 5 }]}>
            Değişiklikleri Kaydet
          </Text>
        </TouchableOpacity>

        <TouchableOpacity
          style={[
            styles.profilePageEditButton,
            { backgroundColor: Colors.secondary, marginTop: 10 },
          ]}
          onPress={() => navigation.goBack()}
          activeOpacity={0.6}
        >
          <Feather name="x" size={24} color={Colors.white} />
          <Text style={[styles.profilePageEditButtonText, { marginLeft: 0 }]}>
            İptal
          </Text>
        </TouchableOpacity>
      </ScrollView>
      <Toolbar selectedMenu={4} />
    </SafeAreaView>
  );
}

export default EditProfileScreen;
