import React from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  Image,
  ActivityIndicator,
} from "react-native";
import { useNavigation } from "expo-router";
import { Feather } from "@expo/vector-icons";
import { Colors, styles } from "@/assets/styles/App.styles";
import Topbar from "@/app/components/Topbar";
import Toolbar from "@/app/components/Toolbar";
import useOrders from "@/app/hooks/queries/useOrders";

const MyOrders = () => {
  const navigation = useNavigation();

  const { data: orders, isLoading, error, refetch } = useOrders();

  const getStatusColor = (status) => {
    switch (status.toLowerCase()) {
      case "delivered":
        return Colors.success;
      case "processing":
        return Colors.warning;
      case "cancelled":
        return Colors.danger;
      default:
        return Colors.secondary;
    }
  };

  const getStatusText = (status) => {
    switch (status) {
      case "DELIVERED":
        return "Teslim Edildi";
      case "PROCESSING":
        return "İşlemde";
      case "PENDING":
        return "Beklemede";
      case "CANCELLED":
        return "İptal Edildi";
      default:
        return status;
    }
  };

  const renderOrderItem = (item, index) => (
    <View
      key={`${item.id}-${index}`}
      style={[styles.orderItemCard, { borderRadius: 12 }]}
    >
      <View style={styles.orderItemImageContainer}>
        <Image
          source={{ uri: item.product_image }}
          style={styles.orderItemImage}
        />
        <View style={styles.orderItemQuantityBadge}>
          <Text style={styles.orderItemQuantityText}>{item.quantity}x</Text>
        </View>
      </View>
      <View style={styles.orderItemInfo}>
        <Text style={styles.orderItemName}>{item.product_name}</Text>
        <Text style={styles.orderItemPrice}>
          {item.product_price} Adım Puan
        </Text>
      </View>
    </View>
  );

  if (isLoading) {
    return (
      <View style={{ flex: 1, justifyContent: "center", alignItems: "center" }}>
        <ActivityIndicator size="large" color={Colors.primary} />
      </View>
    );
  }

  if (error) {
    return (
      <View style={{ flex: 1, justifyContent: "center", alignItems: "center" }}>
        <Text>Hata: {error.message || "Siparişler alınamadı."}</Text>
      </View>
    );
  }

  // orders is expected to be an array from the backend.
  return (
    <View style={styles.MainContainer}>
      <Topbar title="Siparişlerim" />
      <ScrollView
        style={{ width: "100%" }}
        showsVerticalScrollIndicator={false}
      >
        <View style={styles.ordersContainer}>
          {orders && orders.length > 0 ? (
            orders.map((order) => (
              <View key={order.id} style={styles.orderCard}>
                <View style={styles.orderHeader}>
                  <View>
                    <Text style={styles.orderNumber}>
                      Sipariş #{order.order_number}
                    </Text>
                    <Text style={styles.orderDate}>
                      {new Date(order.created_at).toLocaleDateString()}
                    </Text>
                  </View>
                  <View
                    style={[
                      styles.orderStatus,
                      { backgroundColor: getStatusColor(order.status) + "20" },
                    ]}
                  >
                    <Text
                      style={[
                        styles.orderStatusText,
                        { color: getStatusColor(order.status) },
                      ]}
                    >
                      {getStatusText(order.status)}
                    </Text>
                  </View>
                </View>
                <View style={styles.orderItems}>
                  {order.items &&
                    order.items.map((item, index) =>
                      renderOrderItem(item, index)
                    )}
                </View>
                <View style={styles.orderFooter}>
                  <Text style={styles.orderTotal}>
                    Toplam:{" "}
                    {order.items &&
                      order.items
                        .map((item) => item.product_price)
                        .reduce((acc, curr) => acc + curr, 0)}{" "}
                    Adım Puan
                  </Text>
                </View>
              </View>
            ))
          ) : (
            <Text>Henüz siparişiniz bulunmamaktadır.</Text>
          )}
        </View>
      </ScrollView>
      <Toolbar />
    </View>
  );
};

export default MyOrders;
