// app/profile/profile.js
import React, { useState, useCallback, useEffect } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  Image,
  ActivityIndicator,
  Alert,
} from "react-native";
import { Feather } from "@expo/vector-icons";
import Toolbar from "@/app/components/Toolbar";
import { useNavigation, useFocusEffect } from "expo-router";
import { styles, Colors } from "@/assets/styles/App.styles";
import Toast from "@/app/components/common/ToastMessage";
import * as Clipboard from "expo-clipboard";
import Topbar from "@/app/components/Topbar";
import { useUserProfile } from "@/app/hooks/queries/useProfile";
import { useUserPoints } from "@/app/hooks/queries/useUserPoints";
import { useReferralCount } from "@/app/hooks/queries/useReferralCount";
import { useAdvertisements } from "@/app/hooks/queries/useAdvertisements";
import { getImageUrl } from "@/app/utils/imageUtils";
import NetInfo from "@react-native-community/netinfo";
import AsyncStorage from "@react-native-async-storage/async-storage";

import { AdWatch } from "@/app/pages/profile/components/AdWatch";
import { ReferralSection } from "@/app/pages/profile/components/ReferralSection";
import { SettingsSection } from "@/app/pages/profile/components/SettingsSection";

const ProfilePage = () => {
  const navigation = useNavigation();
  const {
    data: user,
    isLoading,
    error,
    refetch: refetchProfile,
  } = useUserProfile();
  const {
    data: points,
    isLoading: pointsLoading,
    error: pointsError,
    refetch: refetchPoints,
  } = useUserPoints();
  const {
    data: referralData,
    isLoading: referralLoading,
    error: referralError,
    refetch: refetchReferral,
  } = useReferralCount();

  // 광고 데이터를 가져오기 위한 hook
  const {
    data: advertisements,
    isLoading: adsLoading,
    error: adsError,
    refetch: refetchAds,
  } = useAdvertisements();

  const [copiedText, setCopiedText] = useState("");
  const [imageLoadError, setImageLoadError] = useState(false);
  const [isOffline, setIsOffline] = useState(false);
  const [retryCount, setRetryCount] = useState(0);
  const [cachedUserData, setCachedUserData] = useState(null);

  // 네트워크 상태 모니터링
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      const isConnected =
        state.isConnected && state.isInternetReachable !== false;
      setIsOffline(!isConnected);

      // 연결이 복구되었을 때만 데이터를 새로고침하고, retryCount 증가는 제거
      if (isConnected && retryCount > 0) {
        refetchAllData();
      }
    });

    // 초기 상태 확인
    NetInfo.fetch().then((state) => {
      setIsOffline(!(state.isConnected && state.isInternetReachable !== false));
    });

    // 캐시된 사용자 데이터 로드
    loadCachedUserData();

    return () => unsubscribe();
  }, [retryCount]); // retryCount는 의존성으로 필요하지만 더 이상 자동 증가하지 않음

  // 모든 데이터 새로고침 함수
  const refetchAllData = useCallback(() => {
    refetchProfile().catch((err) => console.warn("프로필 새로고침 실패:", err));
    refetchPoints().catch((err) => console.warn("포인트 새로고침 실패:", err));
    refetchReferral().catch((err) =>
      console.warn("추천인 새로고침 실패:", err)
    );
    refetchAds().catch((err) => console.warn("광고 새로고침 실패:", err));
    // setRetryCount((prev) => prev + 1); // 이 줄을 제거하여 무한 루프 방지
  }, [refetchProfile, refetchPoints, refetchReferral, refetchAds]);

  // 캐시된 사용자 데이터 로드
  const loadCachedUserData = async () => {
    try {
      const userData = await AsyncStorage.getItem("CACHED_USER_DATA");
      if (userData) {
        setCachedUserData(JSON.parse(userData));
      }
    } catch (e) {
      console.warn("캐시된 사용자 정보 로드 실패:", e);
    }
  };

  // 사용자 데이터 캐싱
  useEffect(() => {
    if (user && !isOffline) {
      try {
        AsyncStorage.setItem("CACHED_USER_DATA", JSON.stringify(user));
      } catch (e) {
        console.warn("사용자 정보 캐싱 실패:", e);
      }
    }
  }, [user, isOffline]);

  useFocusEffect(
    useCallback(() => {
      // 처음 포커스될 때와 네트워크 상태 변경 시에만 새로고침하도록 수정
      let isMounted = true;

      const checkAndRefetch = async () => {
        if (!isMounted) return;

        const state = await NetInfo.fetch();
        const isConnected =
          state.isConnected && state.isInternetReachable !== false;

        setIsOffline(!isConnected);

        // 최초 마운트 시에만 리패치하거나 네트워크가 복구되었을 때만 리패치
        if (isConnected && (retryCount === 0 || retryCount === 1)) {
          refetchAllData();
          // 첫 번째 리패치 후 retryCount를 1로 설정하여 불필요한 리패치 방지
          if (retryCount === 0) {
            setRetryCount(1);
          }
        }
      };

      checkAndRefetch();

      return () => {
        isMounted = false;
      };
    }, [refetchAllData, retryCount])
  );

  const copyToClipboard = async () => {
    if (user) {
      await Clipboard.setStringAsync(user.reference_code);
      Toast("Başarılı", "Referans kodu kopyalandı", "success");
    }
  };

  // 수동 재시도 핸들러 - 사용자가 명시적으로 요청할 때만 retryCount 증가
  const handleRetry = () => {
    setRetryCount((prev) => prev + 1);
    NetInfo.fetch().then((state) => {
      const isConnected =
        state.isConnected && state.isInternetReachable !== false;
      setIsOffline(!isConnected);
      if (isConnected) {
        refetchAllData();
      } else {
        Alert.alert(
          "Bağlantı Hatası",
          "İnternet bağlantınızı kontrol edip tekrar deneyin.",
          [{ text: "Tamam", style: "default" }]
        );
      }
    });
  };

  // 오프라인이고 캐시된 데이터가 있는 경우 오프라인 화면 표시
  if (isOffline && cachedUserData) {
    // 캐시된 데이터로 UI 표시
    const offlineUser = cachedUserData;

    return (
      <View style={styles.profilePageContainer}>
        <Topbar title="Profil (Çevrimdışı)" />
        <View
          style={{
            paddingHorizontal: 15,
            paddingTop: 5,
            backgroundColor: "#FFF7E5",
            flexDirection: "row",
            alignItems: "center",
          }}
        >
          <Feather name="wifi-off" size={16} color="#E69500" />
          <Text style={{ marginLeft: 8, color: "#E69500", fontSize: 14 }}>
            Çevrimdışı moddasınız. Bazı özellikler sınırlı olabilir.
          </Text>
        </View>
        <ScrollView
          style={styles.profilePageScrollContainer}
          showsVerticalScrollIndicator={false}
        >
          <View style={styles.profilePageGradient}>
            {/* Profile Header - 캐시된 데이터 사용 */}
            <View style={styles.profilePageHeaderContainer}>
              <Image
                source={{
                  uri: getImageUrl(
                    offlineUser.profile_pic || offlineUser.profile_image
                  ),
                }}
                style={styles.profilePageAvatar}
                onError={() => setImageLoadError(true)}
                defaultSource={require("@/assets/images/default_avatar.png")}
              />
              <Text style={styles.profilePageName}>
                {offlineUser.first_name} {offlineUser.last_name}
              </Text>
            </View>

            {/* 재연결 버튼 */}
            <TouchableOpacity
              onPress={handleRetry}
              style={{
                backgroundColor: Colors.primary,
                padding: 10,
                borderRadius: 8,
                alignItems: "center",
                margin: 15,
              }}
            >
              <Text style={{ color: Colors.white }}>Tekrar Bağlan</Text>
            </TouchableOpacity>

            {/* 최소한의 정보만 표시 */}
            <View style={styles.profilePageInfoContainer}>
              <View style={styles.profilePageInfoItem}>
                <Feather name="mail" size={20} color={Colors.primary} />
                <Text style={styles.profilePageHeaderText}>
                  {offlineUser.email}
                </Text>
              </View>
              {offlineUser.phone && (
                <View style={styles.profilePageInfoItem}>
                  <Feather name="phone" size={20} color={Colors.primary} />
                  <Text style={styles.profilePageHeaderText}>
                    {offlineUser.phone_code} {offlineUser.phone}
                  </Text>
                </View>
              )}
            </View>
          </View>
        </ScrollView>
        <Toolbar selectedMenu={4} />
      </View>
    );
  }

  // 모든 로딩 상태 체크 (프로필, 포인트, referall, 광고)
  if (isLoading || pointsLoading || referralLoading || adsLoading) {
    return (
      <View style={styles.profilePageContainer}>
        <Topbar title="Profil" />
        <View
          style={{ flex: 1, justifyContent: "center", alignItems: "center" }}
        >
          <ActivityIndicator size="large" color={Colors.primary} />
          <Text style={{ marginTop: 10, color: Colors.secondary }}>
            Profil bilgileri yükleniyor...
          </Text>
        </View>
        <Toolbar selectedMenu={4} />
      </View>
    );
  }

  // 네트워크 오류 화면
  if ((error || pointsError || referralError || adsError) && !user) {
    return (
      <View style={styles.profilePageContainer}>
        <Topbar title="Profil" />
        <View
          style={{
            flex: 1,
            justifyContent: "center",
            alignItems: "center",
            padding: 20,
          }}
        >
          <Feather name="alert-circle" size={48} color={Colors.secondary} />
          <Text style={{ marginTop: 20, fontSize: 18, textAlign: "center" }}>
            {error?.message ||
              pointsError?.message ||
              referralError?.message ||
              adsError?.message ||
              "Bilgiler alınamadı."}
          </Text>
          <TouchableOpacity
            style={{
              marginTop: 20,
              backgroundColor: Colors.primary,
              padding: 12,
              borderRadius: 8,
            }}
            onPress={handleRetry}
          >
            <Text style={{ color: Colors.white }}>Tekrar Dene</Text>
          </TouchableOpacity>
        </View>
        <Toolbar selectedMenu={4} />
      </View>
    );
  }

  // 데이터가 없는 경우 (로딩 완료 후에도)
  if (!user) {
    return (
      <View style={styles.profilePageContainer}>
        <Topbar title="Profil" />
        <View
          style={{ flex: 1, justifyContent: "center", alignItems: "center" }}
        >
          <Text>Bilgiler yüklenemedi. Lütfen daha sonra tekrar deneyin.</Text>
          <TouchableOpacity
            style={{
              marginTop: 20,
              backgroundColor: Colors.primary,
              padding: 12,
              borderRadius: 8,
            }}
            onPress={handleRetry}
          >
            <Text style={{ color: Colors.white }}>Tekrar Dene</Text>
          </TouchableOpacity>
        </View>
        <Toolbar selectedMenu={4} />
      </View>
    );
  }

  // Referral count 값을 사용자 객체에 덮어씌워 사용할 수 있습니다.
  const userWithInvites = {
    ...user,
    inviteCount: referralData?.referral_count ?? user.inviteCount,
  };

  const imagePath = user.profile_pic || user.profile_image;

  // 이미지 URL 생성 시 오류 처리 개선
  let imageUri;
  try {
    if (imagePath) {
      if (imagePath.startsWith("http")) {
        imageUri = imagePath;
      } else {
        imageUri = getImageUrl(imagePath);
      }
    } else {
      imageUri = getImageUrl(""); // 기본 이미지 URL
    }
  } catch (e) {
    console.error("프로필 이미지 URL 생성 오류:", e);
    imageUri = getImageUrl(""); // 오류 시 기본 이미지
  }

  // 이미지 로드 오류 시 기본 이미지로 대체
  const handleImageError = () => {
    console.warn("프로필 이미지 로드 실패:", imageUri);
    setImageLoadError(true);
  };

  const displayImageUri = imageLoadError ? getImageUrl("") : imageUri;

  // 광고 데이터를 video_list_order 값으로 정렬 (광고가 있는 경우)
  const sortedAds = advertisements
    ? [...advertisements]
        .sort((a, b) => a.video_list_order - b.video_list_order)
        .slice(0, 5)
    : [];

  return (
    <View style={styles.profilePageContainer}>
      <Topbar title="Profil" />
      {isOffline && (
        <View
          style={{
            paddingHorizontal: 15,
            paddingVertical: 5,
            backgroundColor: "#FFF7E5",
            flexDirection: "row",
            alignItems: "center",
          }}
        >
          <Feather name="wifi-off" size={16} color="#E69500" />
          <Text style={{ marginLeft: 8, color: "#E69500", fontSize: 14 }}>
            Çevrimdışı moddasınız. Bazı özellikler sınırlı olabilir.
          </Text>
        </View>
      )}
      <ScrollView
        style={styles.profilePageScrollContainer}
        showsVerticalScrollIndicator={false}
      >
        <View style={styles.profilePageGradient}>
          {/* Profile Header */}
          <View style={styles.profilePageHeaderContainer}>
            <Image
              source={{ uri: displayImageUri }}
              style={styles.profilePageAvatar}
              onError={handleImageError}
              defaultSource={require("@/assets/images/default_avatar.png")}
            />
            <Text style={styles.profilePageName}>
              {user.first_name} {user.last_name}
            </Text>
          </View>

          {/* Profile Info */}
          <View style={styles.profilePageInfoContainer}>
            <View style={styles.profilePageInfoItem}>
              <Feather name="mail" size={20} color={Colors.primary} />
              <Text style={styles.profilePageHeaderText}>{user.email}</Text>
            </View>
            <View style={styles.profilePageInfoItem}>
              <Feather name="phone" size={20} color={Colors.primary} />
              <Text style={styles.profilePageHeaderText}>
                {user.phone_code} {user.phone}
              </Text>
            </View>
            <View style={styles.profilePageInfoItem}>
              <Text style={styles.profilePageInfoTextQuestions}>
                Günlük Hedef:
              </Text>
              <Text style={styles.profilePageQuestionInfoText}>
                {user.daily_step_goal} Adım
              </Text>
            </View>
          </View>

          {/* Edit Profile Button */}
          <TouchableOpacity
            onPress={() => navigation.navigate("pages/profile/editProfile")}
            activeOpacity={0.6}
            style={styles.profilePageEditButton}
          >
            <Feather name="edit" size={20} color={Colors.white} />
            <Text style={styles.profilePageEditButtonText}>
              Profili Düzenle
            </Text>
          </TouchableOpacity>

          {/* Referral Section */}
          <ReferralSection
            user={userWithInvites}
            points={points}
            copyToClipboard={copyToClipboard}
          />

          {/* Ad Watch Section */}
          <View style={styles.profilePageInfoContainer}>
            <Text style={styles.ReferanceCodeTextTitle}>
              Reklam izle, adım puan kazan
            </Text>
            {sortedAds.length > 0 ? (
              sortedAds.map((ad) => <AdWatch ad={ad} key={ad.id} />)
            ) : (
              <Text
                style={{
                  padding: 10,
                  textAlign: "center",
                  color: Colors.secondary,
                }}
              >
                {isOffline
                  ? "Çevrimdışı modda reklamlar görüntülenemiyor."
                  : "Şu anda izlenebilir reklam bulunmuyor."}
              </Text>
            )}
          </View>

          {/* Settings Section */}
          <SettingsSection />
        </View>
      </ScrollView>
      <Toolbar selectedMenu={4} />
    </View>
  );
};

export default ProfilePage;
