import React, { useState, useEffect } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  Image,
  Dimensions,
  ActivityIndicator,
} from "react-native";
import { useNavigation, useLocalSearchParams } from "expo-router";
import { Feather } from "@expo/vector-icons";
import { Colors, styles } from "@/assets/styles/App.styles";
import Topbar from "@/app/components/Topbar";
import Toast from "@/app/components/common/ToastMessage";
import Carousel from "react-native-reanimated-carousel";
import Toolbar from "@/app/components/Toolbar";
import { useProductDetail } from "@/app/hooks/queries/useProducts";
import { getImageUrl } from "@/app/utils/imageUtils";
import { useAddCartItem } from "@/app/hooks/queries/useShoppingCart";
import useCartStore from "@/app/store/cartStore";

const ProductDetail = () => {
  const navigation = useNavigation();
  const params = useLocalSearchParams();
  const productId = params.productId;
  const { data: product, isLoading, isError } = useProductDetail(productId);
  const { width } = Dimensions.get("window");

  const [quantity, setQuantity] = useState(1);
  const { mutate: addItemMutation } = useAddCartItem();

  useEffect(() => {
    setQuantity(1);
  }, [product?.id]);

  const cartItem = useCartStore((state) =>
    state.items.find((it) => it.product_id === product?.id)
  );
  const alreadyAdded = cartItem ? cartItem.quantity : 0;
  const productStock = product
    ? product.stock !== undefined
      ? product.stock
      : product.stockCount
    : 0;
  const availableStock = productStock - alreadyAdded;
  const isAddDisabled = availableStock <= 0;

  const handleQuantityChange = (increment) => {
    const newQuantity = quantity + increment;
    if (product && newQuantity >= 1 && newQuantity <= availableStock) {
      setQuantity(newQuantity);
    }
  };

  const addToCart = () => {
    if (!product) return;
    if (quantity < 1 || quantity > availableStock) {
      Toast("Geçersiz miktar", "", "danger");
      return;
    }
    const cartItemData = {
      id: product.id,
      product_id: product.id,
      quantity,
      name: product.name,
      thumbnail:
        product.thumbnail ||
        (product.images && product.images.length > 0
          ? product.images[0].startsWith("http")
            ? product.images[0]
            : getImageUrl(product.images[0])
          : product.image_url
          ? product.image_url.startsWith("http")
            ? product.image_url
            : getImageUrl(product.image_url)
          : ""),
      points_required: product.points_required || product.price,
      stockCount:
        product.stock !== undefined ? product.stock : product.stockCount,
    };
    addItemMutation(cartItemData);
    Toast("Ürün sepete Eklendi", "", "success");
  };

  if (isLoading) {
    return (
      <View style={{ flex: 1, justifyContent: "center", alignItems: "center" }}>
        <ActivityIndicator size="large" color={Colors.primary} />
      </View>
    );
  }

  if (isError || !product) {
    return (
      <View style={{ flex: 1, justifyContent: "center", alignItems: "center" }}>
        <Text>Ürün detayları yüklenemedi.</Text>
      </View>
    );
  }

  const images =
    product.images && product.images.length > 0
      ? product.images.map((img) =>
          img.startsWith("http") ? img : getImageUrl(img)
        )
      : product.image_url
      ? [
          product.image_url.startsWith("http")
            ? product.image_url
            : getImageUrl(product.image_url),
        ]
      : [];

  return (
    <View style={styles.MainContainer}>
      <Topbar title="Ürün Detayı" />
      <ScrollView
        style={{ width: "100%" }}
        showsVerticalScrollIndicator={false}
      >
        <View style={styles.productDetailContainer}>
          <View style={{ overflow: "hidden", width: "100%", height: 300 }}>
            {images.length > 1 ? (
              <Carousel
                mode="default"
                width={width}
                height={300}
                data={images}
                snapEnabled={true}
                pagingEnabled={true}
                autoPlayInterval={3000}
                style={{ width: "100%" }}
                renderItem={({ item }) => (
                  <View
                    style={{
                      flex: 1,
                      paddingVertical: 20,
                      justifyContent: "center",
                      alignItems: "center",
                    }}
                  >
                    <Image
                      source={{ uri: item }}
                      style={{
                        width: "100%",
                        height: "100%",
                        borderRadius: 10,
                        resizeMode: "contain",
                      }}
                    />
                  </View>
                )}
              />
            ) : images.length === 1 ? (
              <View
                style={{
                  flex: 1,
                  paddingVertical: 20,
                  justifyContent: "center",
                  alignItems: "center",
                }}
              >
                <Image
                  source={{ uri: images[0] }}
                  style={{
                    width: "100%",
                    height: "100%",
                    borderRadius: 10,
                    resizeMode: "contain",
                  }}
                />
              </View>
            ) : (
              <View
                style={{
                  width: "100%",
                  height: 300,
                  justifyContent: "center",
                  alignItems: "center",
                }}
              >
                <Text>Görüntü bulunamadı.</Text>
              </View>
            )}
          </View>
          <View style={styles.productDetailInfo}>
            <Text style={styles.productDetailName}>{product.name}</Text>
            <Text style={styles.productDetailPrice}>
              {product.price} Adım Puan
            </Text>
            <Text style={styles.productDetailSectionTitle}>
              Ürün Açıklaması
            </Text>
            <Text style={styles.productDetailDescription}>
              {product.description}
            </Text>
          </View>
        </View>
      </ScrollView>
      <View style={styles.productDetailActionBar}>
        <View style={styles.productDetailQuantity}>
          <TouchableOpacity
            onPress={() => handleQuantityChange(-1)}
            style={styles.productDetailQuantityButton}
          >
            <Feather name="minus" size={20} color={Colors.primary} />
          </TouchableOpacity>
          <Text style={styles.productDetailQuantityText}>{quantity}</Text>
          <TouchableOpacity
            onPress={() => handleQuantityChange(1)}
            style={styles.productDetailQuantityButton}
          >
            <Feather name="plus" size={20} color={Colors.primary} />
          </TouchableOpacity>
        </View>
        <TouchableOpacity
          style={[
            styles.productDetailAddToCart,
            isAddDisabled && { opacity: 0.5 },
          ]}
          onPress={addToCart}
          disabled={isAddDisabled}
        >
          <Feather name="shopping-cart" size={20} color={Colors.white} />
          <Text style={styles.productDetailAddToCartText}>
            {isAddDisabled ? "Stok Doldu" : "Sepete Ekle"}
          </Text>
        </TouchableOpacity>
      </View>
      <Toolbar />
    </View>
  );
};

export default ProductDetail;
