import React, { useCallback, useState, useEffect } from "react";
import {
  View,
  Text,
  ScrollView,
  TouchableOpacity,
  Image,
  LayoutAnimation,
  ActivityIndicator,
} from "react-native";
import { Feather } from "@expo/vector-icons";
import Toolbar from "@/app/components/Toolbar";
import { useNavigation } from "expo-router";
import { styles, Colors } from "@/assets/styles/App.styles";
import Topbar from "@/app/components/Topbar";
import Toast from "@/app/components/common/ToastMessage";
import { useProducts, useCategories } from "@/app/hooks/queries/useProducts";
import { useAddCartItem } from "@/app/hooks/queries/useShoppingCart";
import useCartStore from "@/app/store/cartStore";

const ShopPage = () => {
  const navigation = useNavigation();
  const [isMounted, setIsMounted] = useState(true);

  // 컴포넌트 마운트/언마운트 추적
  useEffect(() => {
    setIsMounted(true);
    return () => {
      setIsMounted(false);
    };
  }, []);

  // useProducts returns product data, loading status, and error.
  const {
    data: products = [],
    isLoading: loadingProducts,
    error: productsError,
  } = useProducts();

  const {
    data: categories = [],
    isLoading: loadingCategories,
    error: categoriesError,
  } = useCategories();

  const { mutate: addItemMutation } = useAddCartItem();

  // Identify the category IDs present in the product list and filter the full list of categories to include only those that are used by the products.
  const dynamicCategories = React.useMemo(() => {
    if (!products || !categories) return [{ id: 0, name: "Tüm Kategoriler" }];

    try {
      const productCategoryIds = new Set(
        products.map((item) => String(item?.category_id || ""))
      );
      const filteredCategories = categories.filter((cat) =>
        cat && cat.id ? productCategoryIds.has(String(cat.id)) : false
      );
      return [{ id: 0, name: "Tüm Kategoriler" }, ...filteredCategories];
    } catch (error) {
      console.error("Error creating dynamic categories:", error);
      return [{ id: 0, name: "Tüm Kategoriler" }];
    }
  }, [products, categories]);

  // Set the default value to 0 (All Categories).
  const [selectedCategoryId, setSelectedCategoryId] = useState(0);

  const handleCategoryPress = useCallback(
    (categoryId) => {
      if (!isMounted) return;

      try {
        LayoutAnimation.configureNext(LayoutAnimation.Presets.easeInEaseOut);
        setSelectedCategoryId(categoryId);
      } catch (error) {
        console.error("Error handling category press:", error);
      }
    },
    [isMounted]
  );

  // Convert product data to match the zustand store and pass it.
  const handleAddToCart = useCallback(
    (product) => {
      const cartItem = useCartStore
        .getState()
        .items.find((ci) => ci.product_id === product.id);
      const currentQuantity = cartItem ? cartItem.quantity : 0;
      const productStock =
        product.stock !== undefined ? product.stock : product.stockCount;
      const availableStock = productStock - currentQuantity;

      if (availableStock <= 0) {
        Toast("Stok Doldu", "Bu ürünün stoğu doldu.", "danger");
        return;
      }
      addItemMutation(product);
      Toast("Sepete Eklendi", "", "success");
    },
    [addItemMutation]
  );

  const ShopItem = ({ item }) => {
    const cartItem = useCartStore((state) =>
      state.items.find((ci) => ci.product_id === item.id)
    );
    const currentQuantity = cartItem ? cartItem.quantity : 0;
    const productStock =
      item.stock !== undefined ? item.stock : item.stockCount;
    const availableStock = productStock - currentQuantity;
    const isButtonDisabled = availableStock <= 0;

    return (
      <TouchableOpacity
        activeOpacity={0.6}
        onPress={() =>
          navigation.navigate("pages/shop/productDetail", {
            productId: item.id,
          })
        }
        style={styles.shopItemCard}
      >
        <Image
          source={{ uri: item.thumbnail }}
          style={[styles.shopItemImage, isButtonDisabled && { opacity: 0.5 }]}
        />
        <Text
          style={styles.shopItemName}
          numberOfLines={1}
          ellipsizeMode="tail"
        >
          {item.name}
        </Text>
        <Text style={styles.shopItemPrice}>
          {item.points_required} Adım Puan
        </Text>
        <TouchableOpacity
          onPress={() => handleAddToCart(item)}
          disabled={isButtonDisabled}
          style={[styles.addToCartButton, isButtonDisabled && { opacity: 0.5 }]}
        >
          <Text style={styles.addToCartButtonText}>
            {isButtonDisabled ? "Stok Doldu" : "Sepete Ekle"}
          </Text>
        </TouchableOpacity>
      </TouchableOpacity>
    );
  };

  // Filtering: If All Categories are selected, all products, otherwise products matching product.category_id.
  const filteredItems =
    selectedCategoryId === 0
      ? products
      : products.filter((item) => item.category_id === selectedCategoryId);

  if (loadingProducts || loadingCategories) {
    return (
      <ActivityIndicator
        size="large"
        color={Colors.primary}
        style={{ flex: 1, justifyContent: "center", alignItems: "center" }}
      />
    );
  }

  if (productsError || categoriesError) {
    return (
      <Text style={{ textAlign: "center", marginTop: 20 }}>
        {productsError?.message ||
          categoriesError?.message ||
          "Bir hata oluştu!"}
      </Text>
    );
  }

  return (
    <View style={styles.WhiteMainContainer}>
      <Topbar title="Mağaza" />
      <ScrollView style={{ flexGrow: 1 }} showsVerticalScrollIndicator={false}>
        <View style={styles.shopCategoryContainer}>
          <ScrollView
            horizontal
            showsHorizontalScrollIndicator={false}
            style={{ height: "100%" }}
            contentContainerStyle={{ alignItems: "center" }}
          >
            {dynamicCategories.map((category) => (
              <TouchableOpacity
                key={category.id}
                style={[
                  styles.shopCategory,
                  selectedCategoryId === category.id && {
                    backgroundColor: Colors.primary,
                  },
                ]}
                onPress={() => handleCategoryPress(category.id)}
              >
                <Text
                  style={
                    selectedCategoryId === category.id
                      ? { color: Colors.white }
                      : { color: Colors.secondary }
                  }
                >
                  {category.name}
                </Text>
              </TouchableOpacity>
            ))}
          </ScrollView>
        </View>
        <View style={styles.shopItemsGrid}>
          {filteredItems.map((item) => (
            <ShopItem key={item.id} item={item} />
          ))}
        </View>
      </ScrollView>
      <Toolbar selectedMenu={2} />
    </View>
  );
};

export default ShopPage;
