// app/store/authStore.js
import { create } from "zustand";
import { persist } from "zustand/middleware";
import api from "@/app/lib/axios";
import AsyncStorage from "@react-native-async-storage/async-storage";
import dayjs from "dayjs";
import useHealthStore from "@/app/store/healthStore";
import { useEffect } from "react";

// AsyncStorage stores values as strings.
// Therefore, we define an adapter below to handle JSON.stringify when saving values.
const asyncStorageAdapter = {
  getItem: (name) => AsyncStorage.getItem(name),
  setItem: (name, value) => {
    if (typeof value !== "string") {
      value = JSON.stringify(value);
    }
    return AsyncStorage.setItem(name, value);
  },
  removeItem: (name) => AsyncStorage.removeItem(name),
};

const useAuthStore = create(
  persist(
    (set) => ({
      user: null,
      token: null,
      isLoading: false,
      error: null,

      initAuth: async () => {
        try {
          const state = await asyncStorageAdapter.getItem("auth-storage");
          if (state) {
            const { state: persistedState } = JSON.parse(state);
            if (persistedState.token) {
              set(persistedState);
            }
          }
        } catch (err) {
          console.error("initAuth error:", err);
        }
      },

      login: async (username, password) => {
        set({ isLoading: true, error: null });
        try {
          const payload = { username, password };
          const response = await api.post("/users/login", payload);
          const { access_token, user } = response.data;
          const updatedUser = user.id
            ? user
            : { ...user, id: user._id || user.user_id };

          // 로그인 전에 기존 건강 데이터 삭제
          await useHealthStore.getState().resetStore();

          // AsyncStorage에서 관련 키들을 추가로 삭제(혹은 확인)
          await AsyncStorage.multiRemove([
            "DAILY_STEP",
            "DAILY_DATE",
            "STEP_HISTORY_KEY",
            "PENDING_STEPS_SYNC",
            "PENDING_STEPS_INCREMENTS",
          ]);

          // 새로운 인증정보 저장
          await AsyncStorage.setItem("token", access_token);
          await AsyncStorage.setItem("user", JSON.stringify(updatedUser));

          // USER_ID를 명시적으로 저장 (걸음수 저장에 필요)
          if (updatedUser.id) {
            await AsyncStorage.setItem("USER_ID", updatedUser.id.toString());
            console.log("User ID stored in AsyncStorage:", updatedUser.id);
          }

          set({ user: updatedUser, token: access_token, isLoading: false });
          return true;
        } catch (error) {
          set({
            error: error.response?.data?.message || "Login failed",
            isLoading: false,
          });
          return false;
        }
      },

      logout: async () => {
        try {
          // 로그아웃 전에 동기화
          if (useAuthStore.getState().token) {
            await useHealthStore.getState().syncSteps();
          }

          // 건강 데이터 리셋
          await useHealthStore.getState().resetStore();

          // 인증 데이터 제거
          await AsyncStorage.multiRemove([
            "token",
            "user",
            "USER_ID",
            "PENDING_STEPS_SYNC",
            "PENDING_STEPS_INCREMENTS",
          ]);
          set({ user: null, token: null });
        } catch (error) {
          console.error("Error during logout:", error);
        }
      },

      setToken: (token) => set({ token }),
    }),
    {
      name: "auth-storage", // AsyncStorage에 저장될 key
      storage: asyncStorageAdapter, // 위에서 정의한 어댑터 사용
    }
  )
);

export const persistStepCount = async () => {
  try {
    const currentSteps = useHealthStore.getState().steps;
    const today = dayjs().format("YYYY-MM-DD");
    await AsyncStorage.setItem("DAILY_STEP", currentSteps.toString());
    await AsyncStorage.setItem("DAILY_DATE", today);
    console.log("Step count persisted:", currentSteps);
  } catch (error) {
    console.error("Error persisting step count:", error);
  }
};

export default useAuthStore;
