// app/store/cartStore.js
import { create } from "zustand";

const useCartStore = create((set, get) => ({
  // The structure of each item: { product_id, quantity, name, thumbnail, points_required, stockCount }
  items: [],

  setCartItems: (items) => set({ items }),

  addItem: (product) =>
    set((state) => {
      // 디버깅 로그 추가
      console.log(
        `Adding product ${product.id} to cart with quantity ${
          product.quantity || 1
        }`
      );

      // If the backend response has a stock key, use that value, otherwise use the existing stockCount
      const productStock =
        product.stock !== undefined ? product.stock : product.stockCount;
      const existing = state.items.find(
        (item) => item.product_id === product.id
      );

      // If 'quantity' is passed, use that value (default is 1)
      const qtyToAdd = product.quantity || 1;

      if (existing) {
        console.log(
          `Product ${product.id} already exists in cart with quantity ${existing.quantity}`
        );

        // 중복 항목일 경우 "설정된 수량"으로 업데이트하도록 변경
        // (기존에는 항상 기존 수량 + 새 수량으로 처리했음)
        let newQuantity;

        // 명시적 업데이트 모드일 경우 (product.updateMode === true)
        if (product.updateMode === true) {
          console.log(`Update mode: Setting quantity to ${qtyToAdd}`);
          newQuantity = qtyToAdd;
        } else {
          // 일반 추가 모드일 경우
          console.log(
            `Add mode: Adding ${qtyToAdd} to existing ${existing.quantity}`
          );
          newQuantity = existing.quantity + qtyToAdd;
        }

        if (newQuantity > productStock) {
          console.log(
            `New quantity ${newQuantity} exceeds stock limit ${productStock}`
          );
          // 재고 초과시 최대 재고량으로 설정
          newQuantity = productStock;
        }

        console.log(`Final quantity for product ${product.id}: ${newQuantity}`);

        return {
          items: state.items.map((item) =>
            item.product_id === product.id
              ? { ...item, quantity: newQuantity }
              : item
          ),
        };
      }

      // 새 항목 추가
      if (qtyToAdd > productStock) {
        console.log(
          `New product quantity ${qtyToAdd} exceeds stock ${productStock}, capping at maximum`
        );
        return {
          items: [
            ...state.items,
            {
              product_id: product.id,
              quantity: productStock,
              name: product.name,
              thumbnail: product.thumbnail,
              points_required: product.points_required,
              stockCount: productStock,
            },
          ],
        };
      }

      console.log(`Adding new product ${product.id} with quantity ${qtyToAdd}`);
      return {
        items: [
          ...state.items,
          {
            product_id: product.id,
            quantity: qtyToAdd,
            name: product.name,
            thumbnail: product.thumbnail,
            points_required: product.points_required,
            stockCount: productStock,
          },
        ],
      };
    }),

  updateItemQuantity: (productId, quantity) =>
    set((state) => ({
      items: state.items.map((item) =>
        item.product_id === productId
          ? {
              ...item,
              // Prevent exceeding stock
              quantity: quantity > item.stockCount ? item.stockCount : quantity,
            }
          : item
      ),
    })),

  removeItem: (productId) =>
    set((state) => ({
      items: state.items.filter((item) => item.product_id !== productId),
    })),

  getCartCount: () =>
    get().items.reduce((total, item) => total + item.quantity, 0),

  // Use the latest stock information from the backend to synchronize the cart
  syncCartStocks: (updatedStocks) =>
    set((state) => ({
      items: state.items.map((item) => {
        const newStock = updatedStocks[item.product_id];
        if (newStock !== undefined) {
          return {
            ...item,
            stockCount: newStock,
            quantity: item.quantity > newStock ? newStock : item.quantity,
          };
        }
        return item;
      }),
    })),
}));

export default useCartStore;
