import { create } from "zustand";
import { createJSONStorage, persist } from "zustand/middleware";
import { Pedometer } from "expo-sensors";
import * as stepsEndpoints from "@/app/api/endpoints/steps";
import dayjs from "dayjs";
import AsyncStorage from "@react-native-async-storage/async-storage";
import useAuthStore from "./authStore";
import NetInfo from "@react-native-community/netinfo";
import {
  checkNetwork,
  resetForNewDay,
  saveStepHistoryToLocal,
  getLocalStepHistory,
  saveStepsForLaterSync,
  getPendingStepsAndClear,
  isUnrealisticStepIncrease,
  recordApiErrorTime,
} from "@/app/utils/healthUtils";
import { addToOfflineQueue } from "@/app/utils/offlineQueueUtils";
import {
  startPedometerSubscription,
  stopPedometerSubscription,
} from "@/app/utils/pedometerUtils";

// 최대 API 호출 실패 횟수
const MAX_FAIL_COUNT = 3;
// API 호출 쿨다운 기간 (밀리초)
const API_COOLDOWN = 5 * 60 * 1000; // 5분

// API 호출 쿨다운 체크
const isInApiCooldown = async () => {
  try {
    const lastErrorTime = await AsyncStorage.getItem("LAST_API_ERROR_TIME");
    if (!lastErrorTime) return false;

    const timeSinceError = Date.now() - parseInt(lastErrorTime);
    return timeSinceError < API_COOLDOWN;
  } catch (error) {
    console.error("Error checking API cooldown:", error);
    return false;
  }
};

// 로컬 걸음수 히스토리 관리
const STEP_HISTORY_KEY = "STEP_HISTORY";

// Add to the store's state
const useHealthStore = create(
  persist(
    (set, get) => ({
      steps: 0,
      lastSyncedSteps: 0,
      isPedometerAvailable: null,
      subscription: null,
      lastSyncDate: null,
      isSyncing: false,
      dailyGoal: 10000,
      failedSyncAttempts: 0,
      isOffline: false,

      setDailyGoal: (goal) => set({ dailyGoal: goal }),

      loadCurrentSteps: async () => {
        try {
          // 새 날짜인지 확인하고 필요시 걸음수 초기화
          const isNewDay = await resetForNewDay();
          if (isNewDay) {
            set({ steps: 0 });
          }

          // 로컬 스토리지에서 걸음수 불러오기 (매번 불러옴)
          const storedSteps = (await AsyncStorage.getItem("DAILY_STEP")) || "0";
          const parsedSteps = parseInt(storedSteps);

          // 로컬에 저장된 걸음수를 항상 먼저 설정 (네트워크 상태와 관계없이)
          set({ steps: parsedSteps });

          // API 쿨다운 상태 확인
          const inCooldown = await isInApiCooldown();
          if (inCooldown) {
            console.log("API 호출 쿨다운 중입니다. 로컬 데이터만 사용합니다.");
            return;
          }

          // 네트워크 연결 확인
          const isNetworkAvailable = await checkNetwork();
          set({ isOffline: !isNetworkAvailable });

          if (!isNetworkAvailable) {
            console.log("No network connection, using local steps");
            return;
          }

          try {
            const data = await stepsEndpoints.getCurrentSteps();
            set({
              steps: data.steps_count,
              lastSyncedSteps: data.steps_count,
              lastSyncDate: new Date(data.updated_at).toISOString(),
              failedSyncAttempts: 0,
            });

            await AsyncStorage.setItem("DAILY_STEP", String(data.steps_count));
            await AsyncStorage.setItem(
              "DAILY_DATE",
              dayjs().format("YYYY-MM-DD")
            );

            // 이 구문 추가: 로컬에 기록 저장
            saveStepHistoryToLocal(
              dayjs().format("YYYY-MM-DD"),
              data.steps_count,
              Math.round(data.steps_count * 0.04),
              Number((data.steps_count * 0.0007).toFixed(2)),
              data.steps_count >= get().dailyGoal
            );
          } catch (error) {
            // 404 오류 처리 - 당일 레코드 없음
            if (error.response?.status === 404) {
              console.log("No steps record found for today");

              // 오늘 레코드가 없고 로컬 걸음수가 있으면, 서버에 새 레코드 생성 시도
              if (parsedSteps > 0) {
                try {
                  // 네트워크 연결 확인
                  const isNetworkAvailable = await checkNetwork();
                  if (!isNetworkAvailable) {
                    console.log(
                      "No network connection, cannot create new record"
                    );
                    return;
                  }

                  // API 쿨다운 상태 확인
                  const inCooldown = await isInApiCooldown();
                  if (inCooldown) {
                    console.log("API 쿨다운 중, 새 레코드 생성 건너뜀");
                    return;
                  }

                  console.log(
                    "Creating new step record with local steps:",
                    parsedSteps
                  );

                  // 새 레코드 생성 (오늘 날짜, 로컬 걸음수)
                  const today = dayjs().format("YYYY-MM-DD");
                  const payload = {
                    record_date: today,
                    steps_count: parsedSteps,
                    points_earned: Math.floor(parsedSteps / 100),
                    calories_burned: Number((parsedSteps * 0.04).toFixed(2)),
                    distance_km: Number((parsedSteps * 0.0007).toFixed(2)),
                    goal_achieved: parsedSteps >= get().dailyGoal,
                    sync_source: "mobile-app",
                  };

                  await stepsEndpoints.sendStepsToDB(parsedSteps);
                  console.log(
                    "Created new step record with local steps:",
                    parsedSteps
                  );

                  // 성공 시 상태 업데이트
                  set({
                    lastSyncedSteps: parsedSteps,
                    lastSyncDate: new Date().toISOString(),
                    failedSyncAttempts: 0,
                  });
                } catch (createError) {
                  console.error(
                    "Error creating new steps record:",
                    createError
                  );

                  // 네트워크 오류 시 쿨다운 활성화
                  if (!createError.response) {
                    await recordApiErrorTime();
                    set({ isOffline: true });
                  }

                  // 오류 발생 횟수 증가
                  set({ failedSyncAttempts: get().failedSyncAttempts + 1 });
                }
              }
            } else {
              console.error("Error loading current steps:", error);

              // 네트워크 오류 시 쿨다운 활성화
              if (!error.response) {
                await recordApiErrorTime();
                set({ isOffline: true });
              }
            }
          }

          // 로컬 기록 항상 저장 (성공/실패 여부와 관계없이)
          saveStepHistoryToLocal(
            dayjs().format("YYYY-MM-DD"),
            parsedSteps,
            Math.round(parsedSteps * 0.04),
            Number((parsedSteps * 0.0007).toFixed(2)),
            parsedSteps >= get().dailyGoal
          );
        } catch (error) {
          console.error("Error in loadCurrentSteps:", error);
          set({ isOffline: true });
        }
      },

      startPedometer: async () => {
        try {
          if (get().subscription) {
            console.log(
              "Pedometer already initialized, skipping duplicate setup"
            );
            return;
          }

          const subscription = await startPedometerSubscription({
            onStepUpdate: ({
              newTotalSteps,
              increment,
              currentSteps,
              storedSteps,
            }) => {
              // Update the state with the new total steps
              set({ steps: newTotalSteps });
              console.log("Pedometer: updated steps", newTotalSteps);
            },
            onSubscriptionSet: (sub) => {
              // Store the subscription in state
              set({ subscription: sub });
            },
          });

          if (!subscription) {
            console.warn("Pedometer subscription was not established.");
          }
        } catch (error) {
          console.error("Error starting pedometer:", error);
        }
      },

      stopPedometer: () => {
        const subscription = get().subscription;
        if (subscription) {
          stopPedometerSubscription(subscription);
          set({ subscription: null });
        }
      },

      syncSteps: async () => {
        try {
          // 이미 동기화 중이면 중복 실행 방지
          if (get().isSyncing) {
            console.log("Step sync already in progress, skipping");
            return;
          }

          set({ isSyncing: true });

          // 쿨다운 상태 확인
          const inCooldown = await isInApiCooldown();
          if (inCooldown) {
            console.log("API 쿨다운 중, 서버 동기화 건너뜀");

            // 로컬 데이터 업데이트는 계속 수행
            const storedSteps =
              (await AsyncStorage.getItem("DAILY_STEP")) || "0";
            const currentSteps = parseInt(storedSteps);

            saveStepHistoryToLocal(
              dayjs().format("YYYY-MM-DD"),
              currentSteps,
              Math.round(currentSteps * 0.04),
              Number((currentSteps * 0.0007).toFixed(2)),
              currentSteps >= get().dailyGoal
            );

            set({ isSyncing: false });
            return;
          }

          // 네트워크 체크
          const isNetworkAvailable = await checkNetwork();
          set({ isOffline: !isNetworkAvailable });

          // 현재 로컬 스텝 가져오기
          const storedSteps = (await AsyncStorage.getItem("DAILY_STEP")) || "0";
          const currentSteps = parseInt(storedSteps);

          // 네트워크 없으면 로컬에 저장만 하고 종료
          if (!isNetworkAvailable) {
            console.log("No network, saving steps locally only:", currentSteps);

            // 로컬 히스토리 업데이트
            saveStepHistoryToLocal(
              dayjs().format("YYYY-MM-DD"),
              currentSteps,
              Math.round(currentSteps * 0.04),
              Number((currentSteps * 0.0007).toFixed(2)),
              currentSteps >= get().dailyGoal
            );

            // 동기화 상태 종료
            set({ isSyncing: false });
            return;
          }

          // 이전에 저장된 마지막 동기화 걸음수
          const prevSyncedSteps = get().lastSyncedSteps;

          // 서버의 현재 걸음수 확인 (DB와 로컬 간 불일치 방지)
          try {
            console.log("Verifying current server step count...");
            const serverData = await stepsEndpoints.getCurrentSteps();
            const serverSteps = serverData.steps_count;

            console.log("Step counts:", {
              localSteps: currentSteps,
              serverSteps: serverSteps,
              lastSyncedSteps: prevSyncedSteps,
            });

            // 서버 걸음수가 로컬보다 많은 경우 (다른 기기에서 업데이트 등)
            if (serverSteps > currentSteps) {
              console.log(
                "Server has more steps than local, updating local count"
              );
              await AsyncStorage.setItem("DAILY_STEP", String(serverSteps));

              // 로컬 히스토리 업데이트
              saveStepHistoryToLocal(
                dayjs().format("YYYY-MM-DD"),
                serverSteps,
                Math.round(serverSteps * 0.04),
                Number((serverSteps * 0.0007).toFixed(2)),
                serverSteps >= get().dailyGoal
              );

              set({
                steps: serverSteps,
                lastSyncedSteps: serverSteps,
                lastSyncDate: new Date().toISOString(),
                failedSyncAttempts: 0,
                isSyncing: false,
                isOffline: false,
              });

              return;
            }

            // 로컬과 서버 값이 같은 경우 - 동기화 필요 없음
            if (serverSteps === currentSteps) {
              console.log("Local and server step counts are already in sync");
              set({
                lastSyncedSteps: currentSteps,
                lastSyncDate: new Date().toISOString(),
                failedSyncAttempts: 0,
                isSyncing: false,
                isOffline: false,
              });
              return;
            }

            // 이 시점에서 localSteps > serverSteps 케이스만 남음
            // 동기화할 걸음수 계산 (로컬 증가분)
            const stepsToSync = currentSteps - serverSteps;

            // 동기화가 필요한지 확인 (100보 미만은 동기화하지 않음)
            if (stepsToSync < 100) {
              console.log(`Not enough steps to sync (${stepsToSync} < 100)`);

              // 로컬 히스토리는 업데이트
              saveStepHistoryToLocal(
                dayjs().format("YYYY-MM-DD"),
                currentSteps,
                Math.round(currentSteps * 0.04),
                Number((currentSteps * 0.0007).toFixed(2)),
                currentSteps >= get().dailyGoal
              );

              set({ isSyncing: false });
              return;
            }

            console.log(`Syncing steps increment to server: ${stepsToSync}`);

            // 서버에 걸음수 증가분 전송
            const pendingSteps = await getPendingStepsAndClear();
            const totalStepsToSync = stepsToSync + pendingSteps;
            console.log("Syncing total steps:", totalStepsToSync);

            // API 호출
            const response = await stepsEndpoints.accumulateSteps(
              totalStepsToSync
            );
            console.log("Steps successfully synced:", totalStepsToSync);

            // 서버에서 받은 최종 걸음수 확인 (검증용)
            const finalServerSteps = response.steps_count;
            console.log(
              "Final server step count after sync:",
              finalServerSteps
            );

            // 서버 값과 로컬 값 동기화
            await AsyncStorage.setItem("DAILY_STEP", String(finalServerSteps));

            // 로컬 히스토리 갱신
            saveStepHistoryToLocal(
              dayjs().format("YYYY-MM-DD"),
              finalServerSteps,
              Math.round(finalServerSteps * 0.04),
              Number((finalServerSteps * 0.0007).toFixed(2)),
              finalServerSteps >= get().dailyGoal
            );

            // 쿨다운 해제 (성공 시)
            await AsyncStorage.removeItem("LAST_API_ERROR_TIME");

            // 성공적으로 동기화 완료
            set({
              steps: finalServerSteps,
              lastSyncedSteps: finalServerSteps,
              lastSyncDate: new Date().toISOString(),
              failedSyncAttempts: 0,
              isSyncing: false,
              isOffline: false,
            });
          } catch (error) {
            // 서버 확인 중 오류 발생
            if (error.response?.status === 404) {
              // 404 오류 처리 (당일 레코드 없음)
              try {
                console.log("No steps record found, creating new record...");

                // 실패 횟수 확인
                const failCount = get().failedSyncAttempts;
                if (failCount >= MAX_FAIL_COUNT) {
                  console.log(
                    `최대 실패 횟수(${MAX_FAIL_COUNT}) 도달, 쿨다운 설정`
                  );
                  await recordApiErrorTime();
                  await addToOfflineQueue(currentSteps);
                  set({
                    isSyncing: false,
                    failedSyncAttempts: 0, // 카운터 리셋
                  });

                  // 로컬 히스토리 업데이트
                  saveStepHistoryToLocal(
                    dayjs().format("YYYY-MM-DD"),
                    currentSteps,
                    Math.round(currentSteps * 0.04),
                    Number((currentSteps * 0.0007).toFixed(2)),
                    currentSteps >= get().dailyGoal
                  );

                  return;
                }

                // 새 레코드 생성 시도 - 로컬 전체 걸음수로 시작
                await stepsEndpoints.sendStepsToDB(currentSteps);
                console.log(
                  "New record created with current steps:",
                  currentSteps
                );

                // 로컬 히스토리 업데이트
                saveStepHistoryToLocal(
                  dayjs().format("YYYY-MM-DD"),
                  currentSteps,
                  Math.round(currentSteps * 0.04),
                  Number((currentSteps * 0.0007).toFixed(2)),
                  currentSteps >= get().dailyGoal
                );

                // 쿨다운 해제 (성공 시)
                await AsyncStorage.removeItem("LAST_API_ERROR_TIME");

                set({
                  lastSyncedSteps: currentSteps,
                  lastSyncDate: new Date().toISOString(),
                  failedSyncAttempts: 0,
                  isSyncing: false,
                  isOffline: false,
                });
              } catch (createError) {
                // 레코드 생성 실패 시
                console.error("Error creating steps record:", createError);
                await addToOfflineQueue(currentSteps);

                // 네트워크 오류인 경우 쿨다운 활성화
                if (!createError.response) {
                  const failures = get().failedSyncAttempts + 1;

                  if (failures >= MAX_FAIL_COUNT) {
                    console.log(
                      `최대 실패 횟수(${MAX_FAIL_COUNT}) 도달, 쿨다운 설정`
                    );
                    await recordApiErrorTime();
                    set({
                      isSyncing: false,
                      failedSyncAttempts: 0,
                      isOffline: true,
                    });
                  } else {
                    set({
                      isSyncing: false,
                      failedSyncAttempts: failures,
                      isOffline: true,
                    });
                  }
                } else {
                  set({
                    isSyncing: false,
                    failedSyncAttempts: get().failedSyncAttempts + 1,
                  });
                }

                // 로컬 히스토리 업데이트
                saveStepHistoryToLocal(
                  dayjs().format("YYYY-MM-DD"),
                  currentSteps,
                  Math.round(currentSteps * 0.04),
                  Number((currentSteps * 0.0007).toFixed(2)),
                  currentSteps >= get().dailyGoal
                );
              }
            } else {
              // 다른 API 오류
              console.error("Error verifying server steps:", error);

              // 실패 횟수 증가
              const failures = get().failedSyncAttempts + 1;

              // 최대 실패 횟수 도달 시 쿨다운 활성화
              if (!error.response && failures >= MAX_FAIL_COUNT) {
                console.log(
                  `최대 실패 횟수(${MAX_FAIL_COUNT}) 도달, 쿨다운 설정`
                );
                await recordApiErrorTime();
                set({
                  isSyncing: false,
                  failedSyncAttempts: 0,
                  isOffline: true,
                });
              } else {
                // 실패 시 pending 저장
                await addToOfflineQueue(currentSteps - prevSyncedSteps);
                set({
                  isSyncing: false,
                  failedSyncAttempts: failures,
                  isOffline: !error.response,
                });
              }

              // 로컬 히스토리는 계속 업데이트
              saveStepHistoryToLocal(
                dayjs().format("YYYY-MM-DD"),
                currentSteps,
                Math.round(currentSteps * 0.04),
                Number((currentSteps * 0.0007).toFixed(2)),
                currentSteps >= get().dailyGoal
              );
            }
          }
        } catch (error) {
          // 전체 함수에 대한 에러 핸들링
          console.error("Error in syncSteps function:", error);

          // 네트워크 오류 시 쿨다운 활성화
          if (!error.response) {
            const failures = get().failedSyncAttempts + 1;

            if (failures >= MAX_FAIL_COUNT) {
              await recordApiErrorTime();
              set({
                isSyncing: false,
                failedSyncAttempts: 0,
                isOffline: true,
              });
            } else {
              set({
                isSyncing: false,
                failedSyncAttempts: failures,
                isOffline: true,
              });
            }
          } else {
            set({
              isSyncing: false,
              failedSyncAttempts: get().failedSyncAttempts + 1,
              isOffline: true,
            });
          }

          // 로컬 히스토리 업데이트 시도
          try {
            const storedSteps =
              (await AsyncStorage.getItem("DAILY_STEP")) || "0";
            const currentSteps = parseInt(storedSteps);

            saveStepHistoryToLocal(
              dayjs().format("YYYY-MM-DD"),
              currentSteps,
              Math.round(currentSteps * 0.04),
              Number((currentSteps * 0.0007).toFixed(2)),
              currentSteps >= get().dailyGoal
            );
          } catch (localError) {
            console.error(
              "Error saving local steps in error handler:",
              localError
            );
          }
        }
      },

      resetStore: async () => {
        try {
          // First, stop any active subscription
          get().stopPedometer();

          // Get all the possible user-specific keys for complete cleanup
          const userId = useAuthStore.getState().user?.id;
          const keysToRemove = [
            "DAILY_STEP",
            "DAILY_DATE",
            "STEP_HISTORY",
            "LAST_HISTORY_FETCH",
            "PENDING_STEPS",
            "LAST_API_ERROR_TIME",
            "LAST_SYNC_ATTEMPT",
            "STEPS_OFFLINE_QUEUE",
            "health-store", // zustand persist storage key
            // Add user-specific variants
            userId ? `DAILY_STEP_${userId}` : null,
            userId ? `DAILY_DATE_${userId}` : null,
            userId ? `STEP_HISTORY_${userId}` : null,
            userId ? `LAST_HISTORY_FETCH_${userId}` : null,
            userId ? `LAST_STEPS_ERROR_TIME_${userId}` : null,
            userId ? `LAST_SYNC_ATTEMPT_${userId}` : null,
            userId ? `STEPS_OFFLINE_QUEUE_${userId}` : null,
          ].filter(Boolean); // Filter out any null values

          // Reset all state
          set({
            steps: 0,
            lastSyncedSteps: 0,
            isPedometerAvailable: null,
            subscription: null,
            lastSyncDate: null,
            isSyncing: false,
            dailyGoal: 10000,
            failedSyncAttempts: 0,
            isOffline: false,
          });

          // Clear AsyncStorage keys
          await AsyncStorage.multiRemove(keysToRemove);

          console.log("Health store completely reset");
        } catch (error) {
          console.error("Error resetting health store:", error);
        }
      },
    }),
    {
      name: "health-store",
      storage: createJSONStorage(() => AsyncStorage),
      partialize: (state) => ({
        steps: state.steps,
        lastSyncedSteps: state.lastSyncedSteps,
        dailyGoal: state.dailyGoal,
        failedSyncAttempts: state.failedSyncAttempts,
        isOffline: state.isOffline,
      }),
    }
  )
);

export default useHealthStore;
