// app/store/notificationStore.js
import { create } from "zustand";
import { fetchNotifications } from "@/app/api/endpoints/notifications";
import useAuthStore from "@/app/store/authStore";

const useNotificationStore = create((set, get) => ({
  notifications: [],
  error: null,
  isLoading: false,

  // Fetch notifications
  fetchNotifications: async () => {
    set({ isLoading: true, error: null });
    try {
      // 토큰 확인
      const token = useAuthStore.getState().token;
      if (!token) {
        console.log("No token available, skipping notifications fetch");
        set({ isLoading: false });
        return [];
      }

      const data = await fetchNotifications();
      if (!data || !Array.isArray(data)) {
        console.warn("Invalid notifications data received:", data);
        set({ notifications: [], isLoading: false });
        return [];
      }

      // Sort notifications in descending order (newest first)
      const sortedData = data.sort(
        (a, b) => new Date(b.created_at) - new Date(a.created_at)
      );
      set({ notifications: sortedData, isLoading: false });
      return sortedData;
    } catch (error) {
      console.warn("Error fetching notifications:", error);

      if (error.response?.status === 404) {
        // 404 에러 시 빈 배열로 설정
        set({ notifications: [], isLoading: false });
        return [];
      } else if (error.response?.status === 401) {
        // 인증 오류 시 로그아웃
        await useAuthStore.getState().logout();
      }

      set({
        error: error.message || "Failed to fetch notifications",
        notifications: [],
        isLoading: false,
      });
      return [];
    }
  },

  // Getters
  getUnreadCount: () => {
    const { notifications } = get();
    return notifications.filter((notification) => notification.read_at === null)
      .length;
  },

  // Setters
  setNotifications: (notifications) => set({ notifications }),
  setError: (error) => set({ error }),
  setLoading: (isLoading) => set({ isLoading }),
}));

export default useNotificationStore;
