/* app/utils/healthUtils.js */
import AsyncStorage from "@react-native-async-storage/async-storage";
import dayjs from "dayjs";
import NetInfo from "@react-native-community/netinfo";

// Check network connectivity
export const checkNetwork = async () => {
  try {
    const state = await NetInfo.fetch();
    return state.isConnected && state.isInternetReachable;
  } catch (error) {
    console.error("Error checking network:", error);
    return false;
  }
};

// Reset steps for a new day
export const resetForNewDay = async () => {
  try {
    const lastDateStr = await AsyncStorage.getItem("DAILY_DATE");
    const today = dayjs().format("YYYY-MM-DD");
    if (lastDateStr && lastDateStr !== today) {
      console.log("New day detected, resetting local step counter");
      await AsyncStorage.setItem("DAILY_STEP", "0");
      await AsyncStorage.setItem("DAILY_DATE", today);
      return true;
    }
    return false;
  } catch (error) {
    console.error("Error checking for new day:", error);
    return false;
  }
};

// Save step history to local storage
export const saveStepHistoryToLocal = async (
  date,
  steps,
  calories,
  distance,
  goalAchieved = false
) => {
  try {
    const STEP_HISTORY_KEY = "STEP_HISTORY";
    const historyStr = await AsyncStorage.getItem(STEP_HISTORY_KEY);
    const history = historyStr ? JSON.parse(historyStr) : {};
    history[date] = { steps, calories, distance, goalAchieved, date };
    await AsyncStorage.setItem(STEP_HISTORY_KEY, JSON.stringify(history));
    console.log(`Saved ${steps} steps for ${date} to local history`);
  } catch (error) {
    console.error("Error saving step history:", error);
  }
};

// Get local step history
export const getLocalStepHistory = async () => {
  try {
    const STEP_HISTORY_KEY = "STEP_HISTORY";
    const historyStr = await AsyncStorage.getItem(STEP_HISTORY_KEY);
    return historyStr ? JSON.parse(historyStr) : {};
  } catch (error) {
    console.error("Error loading step history:", error);
    return {};
  }
};

// Save pending steps for later syncing
export const saveStepsForLaterSync = async (increment) => {
  try {
    const pendingSteps = (await AsyncStorage.getItem("PENDING_STEPS")) || "0";
    const newPendingSteps = parseInt(pendingSteps) + increment;
    await AsyncStorage.setItem("PENDING_STEPS", newPendingSteps.toString());
    console.log(
      "Saved steps for later sync:",
      increment,
      "total pending:",
      newPendingSteps
    );
  } catch (error) {
    console.error("Error saving pending steps:", error);
  }
};

// Get pending steps and clear them
export const getPendingStepsAndClear = async () => {
  try {
    const pendingSteps = (await AsyncStorage.getItem("PENDING_STEPS")) || "0";
    const steps = parseInt(pendingSteps);
    if (steps > 0) {
      await AsyncStorage.setItem("PENDING_STEPS", "0");
    }
    return steps;
  } catch (error) {
    console.error("Error getting pending steps:", error);
    return 0;
  }
};

// Detect unrealistic step increases
export const isUnrealisticStepIncrease = (increment, timeDiff) => {
  const MAX_STEPS_PER_SECOND = 5;
  if (timeDiff < 1000 && increment > 5) {
    console.warn(
      `Unrealistic step increase detected: ${increment} steps in ${timeDiff}ms`
    );
    return true;
  }
  const secondsElapsed = timeDiff / 1000;
  const maxRealisticIncrease = Math.ceil(secondsElapsed * MAX_STEPS_PER_SECOND);
  if (increment > maxRealisticIncrease) {
    console.warn(
      `Possible sensor error: ${increment} steps in ${secondsElapsed.toFixed(
        1
      )}s (max realistic: ${maxRealisticIncrease})`
    );
    return true;
  }
  return false;
};

// Record API error time for cooldown
export const recordApiErrorTime = async () => {
  try {
    await AsyncStorage.setItem("LAST_API_ERROR_TIME", Date.now().toString());
  } catch (error) {
    console.error("Error recording API error time:", error);
  }
};
