// app/utils/imageUtils.js
import Constants from "expo-constants";
import { Platform } from "react-native";

/**
 * Generate a full URL to access an image stored in AWS S3/CloudFront.
 *
 * This function is used in various places where an image URL is needed,
 * such as the default profile image for users without a registered photo
 * or in product logic.
 *
 * @param {string} path - The actual image path.
 * @param {string} [defaultPath="profile/default_avatar.png"] - The default image path to use if no image is available.
 * @returns {string} - The complete image URL.
 */
export const getImageUrl = (
  path,
  defaultPath = "profile/default_avatar.png"
) => {
  try {
    // 빈 문자열, null, undefined 체크
    if (!path || path === "") {
      // 개발 환경에서만 로그를 출력합니다
      if (__DEV__) {
        console.log("이미지 경로가 없어 기본 이미지 사용:", defaultPath);
      }
      const defaultUrl = `https://adimsayar.easerver.net/${defaultPath}`;
      return defaultUrl;
    }

    // 이미 http:// 또는 https://로 시작하는 전체 URL인지 확인
    if (path.startsWith("http://") || path.startsWith("https://")) {
      // URL에 특수 문자가 있는지 확인
      if (path.includes(" ") || path.includes("%") || path.includes("&")) {
        // 앞서 인코딩된 URL인지 확인 (중복 인코딩 방지)
        if (!path.includes("%25")) {
          const encodedUrl = encodeURI(path);
          // 개발 환경에서만 로그를 출력합니다
          if (__DEV__) {
            console.log("특수 문자가 있는 URL을 인코딩:", encodedUrl);
          }
          return encodedUrl;
        }
      }
      return path; // 이미 완전한 URL이면 그대로 반환
    }

    // 안드로이드에서 특수 문자 처리를 더 엄격하게
    let encodedPath = path;
    if (Platform.OS === "android") {
      // URL 인코딩 적용 (공백 및 특수문자 처리)
      encodedPath = encodeURIComponent(path).replace(/%2F/g, "/"); // 슬래시는 인코딩하지 않음
    } else {
      // iOS는 덜 엄격한 인코딩 적용
      encodedPath = path.replace(/\s+/g, "%20"); // 공백만 인코딩
    }

    // 개발 환경에서만 로그를 출력합니다
    if (__DEV__) {
      console.log(`이미지 경로 인코딩: ${path} → ${encodedPath}`);
    }

    // 전체 URL이 아닌 경우에만 도메인 추가
    return `https://adimsayar.easerver.net/${encodedPath}`;
  } catch (error) {
    console.warn("이미지 URL 생성 오류:", error);
    // 오류 발생 시 기본 이미지 반환
    return `https://adimsayar.easerver.net/${defaultPath}`;
  }
};
