/* app/utils/offlineQueueUtils.js */
import AsyncStorage from "@react-native-async-storage/async-storage";
import useAuthStore from "@/app/store/authStore";
import dayjs from "dayjs";

// 이전 큐 버전과의 호환성을 위한 상수
const QUEUE_VERSION = "2";
const MAX_QUEUE_SIZE = 50; // 최대 큐 크기 제한

// Helper to get user id from auth store
const getUserId = () => {
  const user = useAuthStore.getState().user;
  return user?.id || "";
};

// Generate the offline queue storage key based on user id
export const getOfflineQueueKey = () => {
  const userId = getUserId();
  const baseKey = userId
    ? `STEPS_OFFLINE_QUEUE_${userId}`
    : "STEPS_OFFLINE_QUEUE";
  return `${baseKey}_V${QUEUE_VERSION}`;
};

// Retrieve the offline queue from AsyncStorage
export const getOfflineQueue = async () => {
  try {
    const queueData = await AsyncStorage.getItem(getOfflineQueueKey());

    // 새 버전의 큐 확인
    if (queueData) {
      return JSON.parse(queueData);
    }

    // 이전 버전 큐가 있는지 확인 (마이그레이션을 위해)
    const userId = getUserId();
    const legacyKey = userId
      ? `STEPS_OFFLINE_QUEUE_${userId}`
      : "STEPS_OFFLINE_QUEUE";

    if (legacyKey !== getOfflineQueueKey()) {
      const legacyQueueData = await AsyncStorage.getItem(legacyKey);
      if (legacyQueueData) {
        const legacyQueue = JSON.parse(legacyQueueData);
        console.log(
          `Migrating ${legacyQueue.length} items from legacy queue to new format`
        );

        // 이전 큐를 새 형식으로 변환
        await AsyncStorage.setItem(
          getOfflineQueueKey(),
          JSON.stringify(legacyQueue)
        );

        // 이전 큐 삭제
        await AsyncStorage.removeItem(legacyKey);

        return legacyQueue;
      }
    }

    return [];
  } catch (error) {
    console.error("Error reading offline queue:", error);
    // 오류 시 안전하게 빈 배열 반환
    return [];
  }
};

// Add steps to the offline queue; optionally update local history via a callback
export const addToOfflineQueue = async (
  steps,
  updateLocalStepHistoryCallback
) => {
  try {
    if (!steps || steps <= 0) {
      console.log("Skipping adding 0 or negative steps to offline queue");
      return;
    }

    const queue = await getOfflineQueue();

    // 중복 항목 확인 (타임스탬프 기준 1초 이내)
    const now = new Date();
    const isDuplicate = queue.some((item) => {
      const itemTime = new Date(item.timestamp);
      return Math.abs(now - itemTime) < 1000 && item.steps === steps;
    });

    if (isDuplicate) {
      console.log("Duplicate entry detected, skipping");
      return;
    }

    // 큐 크기 제한 확인
    if (queue.length >= MAX_QUEUE_SIZE) {
      console.log(
        `Queue exceeds maximum size (${MAX_QUEUE_SIZE}), removing oldest entry`
      );
      queue.shift(); // 가장 오래된 항목 제거
    }

    // 새 항목 추가
    const newEntry = {
      steps,
      timestamp: now.toISOString(),
      date: dayjs().format("YYYY-MM-DD"), // 날짜 정보 추가
    };

    queue.push(newEntry);
    await AsyncStorage.setItem(getOfflineQueueKey(), JSON.stringify(queue));
    console.log("Steps added to offline queue:", steps);

    if (typeof updateLocalStepHistoryCallback === "function") {
      await updateLocalStepHistoryCallback(steps);
    }
  } catch (error) {
    console.error("Error adding to offline queue:", error);
  }
};

// 큐 항목을 처리완료로 표시하거나 제거
export const markItemProcessed = async (timestamp) => {
  try {
    const queue = await getOfflineQueue();
    const updatedQueue = queue.filter((item) => item.timestamp !== timestamp);

    if (updatedQueue.length < queue.length) {
      await AsyncStorage.setItem(
        getOfflineQueueKey(),
        JSON.stringify(updatedQueue)
      );
      console.log(`Removed processed item (${timestamp}) from queue`);
    }
  } catch (error) {
    console.error("Error marking item as processed:", error);
  }
};

// 큐 전체 데이터 비우기
export const clearOfflineQueue = async () => {
  try {
    await AsyncStorage.setItem(getOfflineQueueKey(), JSON.stringify([]));
    console.log("Offline queue cleared");
  } catch (error) {
    console.error("Error clearing offline queue:", error);
  }
};

// 오래된 큐 항목 정리 (30일 이상 지난 항목)
export const cleanupOldQueueItems = async () => {
  try {
    const queue = await getOfflineQueue();
    const thirtyDaysAgo = dayjs().subtract(30, "day");

    const filteredQueue = queue.filter((item) => {
      const itemDate = dayjs(item.timestamp);
      return itemDate.isAfter(thirtyDaysAgo);
    });

    if (filteredQueue.length < queue.length) {
      await AsyncStorage.setItem(
        getOfflineQueueKey(),
        JSON.stringify(filteredQueue)
      );
      console.log(
        `Cleaned up ${queue.length - filteredQueue.length} old queue items`
      );
    }
  } catch (error) {
    console.error("Error cleaning up old queue items:", error);
  }
};
