import { Pedometer } from "expo-sensors";
import AsyncStorage from "@react-native-async-storage/async-storage";
import dayjs from "dayjs";
import {
  isUnrealisticStepIncrease,
  checkNetwork,
  recordApiErrorTime,
} from "./healthUtils";
import { addToOfflineQueue } from "./offlineQueueUtils";
import * as stepsEndpoints from "@/app/api/endpoints/steps";

/**
 * Starts the pedometer subscription and handles sensor events with debounce.
 * @param {Object} options - Configuration options.
 * @param {Function} options.onStepUpdate - Callback function invoked on step update; receives an object with newTotalSteps, increment, currentSteps, and storedSteps.
 * @param {Function} options.onSubscriptionSet - Callback invoked when subscription is set, receives the subscription object.
 * @returns {Promise<Object>} The pedometer subscription object.
 */
export const startPedometerSubscription = async (options = {}) => {
  try {
    const available = await Pedometer.isAvailableAsync();
    if (!available) {
      console.error("Pedometer not available on this device.");
      return null;
    }

    let lastStepCount = null;
    let lastUpdateTime = Date.now();
    const debounceTime = 1000; // 1 second debounce
    let consecutiveZeroIncrements = 0;
    let syncFailCount = 0;

    console.log("Starting new pedometer subscription");
    const subscription = Pedometer.watchStepCount(async (result) => {
      const currentSteps = result.steps;
      const now = Date.now();

      // Initialize baseline on first reading
      if (lastStepCount === null) {
        lastStepCount = currentSteps;
        lastUpdateTime = now;
        console.log(
          "Initialized pedometer with baseline step count:",
          currentSteps
        );
        return;
      }

      // Debounce rapid updates
      if (now - lastUpdateTime < debounceTime) {
        return;
      }

      const increment = Math.max(0, currentSteps - lastStepCount);
      const timeSinceLastUpdate = now - lastUpdateTime;

      // Log zero increments for debugging
      if (increment === 0) {
        consecutiveZeroIncrements++;
        if (consecutiveZeroIncrements >= 5) {
          console.log(
            `Received ${consecutiveZeroIncrements} consecutive zero increments from pedometer`
          );
          consecutiveZeroIncrements = 0;
        }
        lastStepCount = currentSteps;
        lastUpdateTime = now;
        return;
      } else {
        consecutiveZeroIncrements = 0;
      }

      // Skip unrealistic step increases
      if (
        increment > 0 &&
        isUnrealisticStepIncrease(increment, timeSinceLastUpdate)
      ) {
        console.log("Skipping unrealistic step increase");
        lastStepCount = currentSteps;
        lastUpdateTime = now;
        return;
      }

      if (increment > 0) {
        // Get current stored steps
        const storedStepsStr =
          (await AsyncStorage.getItem("DAILY_STEP")) || "0";
        const storedSteps = parseInt(storedStepsStr, 10);
        const newTotalSteps = storedSteps + increment;

        console.log("Step update:", {
          lastCount: lastStepCount,
          currentCount: currentSteps,
          increment,
          storedSteps,
          newTotal: newTotalSteps,
          timeSinceLastUpdate: Math.round(timeSinceLastUpdate / 1000) + "s",
        });

        // Update local storage with new total steps and current date
        await AsyncStorage.setItem("DAILY_STEP", String(newTotalSteps));
        await AsyncStorage.setItem("DAILY_DATE", dayjs().format("YYYY-MM-DD"));

        // Invoke callback to notify external listener
        if (typeof options.onStepUpdate === "function") {
          options.onStepUpdate({
            newTotalSteps,
            increment,
            currentSteps,
            storedSteps,
          });
        }

        // Check network and attempt to sync steps - with improved error handling
        const isNetworkAvailable = await checkNetwork();
        if (!isNetworkAvailable) {
          console.log("No network connection, saving steps for later sync");
          await addToOfflineQueue(increment);
        } else {
          try {
            // Check if we're not in a cooldown period before attempting to sync
            const lastErrorTime = await AsyncStorage.getItem(
              "LAST_API_ERROR_TIME"
            );
            const cooldownPeriod = 5 * 60 * 1000; // 5 minutes
            const isInCooldown =
              lastErrorTime &&
              Date.now() - parseInt(lastErrorTime) < cooldownPeriod;

            if (isInCooldown) {
              console.log(
                "In API cooldown period, saving steps for later sync"
              );
              await addToOfflineQueue(increment);
            } else {
              // Verify server's current steps before sending increment
              try {
                const serverData = await stepsEndpoints.getCurrentSteps();
                const serverSteps = serverData.steps_count;

                // If server has more steps than our local total, something's wrong
                if (serverSteps > newTotalSteps) {
                  console.log(
                    "Server has more steps than local, updating local count"
                  );
                  await AsyncStorage.setItem("DAILY_STEP", String(serverSteps));

                  // Notify external listeners of the sync
                  if (typeof options.onStepUpdate === "function") {
                    options.onStepUpdate({
                      newTotalSteps: serverSteps,
                      increment: 0, // No real increment, just syncing
                      currentSteps: serverSteps,
                      storedSteps: serverSteps,
                    });
                  }

                  syncFailCount = 0;
                  lastStepCount = currentSteps;
                  lastUpdateTime = now;
                  return;
                }

                // Only sync the difference between server and our new total
                const stepsToSync = newTotalSteps - serverSteps;

                if (stepsToSync > 0) {
                  console.log(`Syncing ${stepsToSync} steps to server`);
                  const response = await stepsEndpoints.accumulateSteps(
                    stepsToSync
                  );
                  console.log("Successfully synced steps:", response);
                  syncFailCount = 0;
                } else {
                  console.log(
                    "No steps to sync (server already has latest count)"
                  );
                }
              } catch (serverCheckError) {
                // If 404, create a new record
                if (serverCheckError.response?.status === 404) {
                  console.log(
                    "No steps record found for today, creating new record"
                  );
                  await stepsEndpoints.sendStepsToDB(newTotalSteps);
                  console.log("New steps record created successfully");
                  syncFailCount = 0;
                } else {
                  throw serverCheckError; // Re-throw for general error handling
                }
              }
            }
          } catch (syncError) {
            syncFailCount++;
            console.warn(
              `Error syncing steps (fail #${syncFailCount}), saving for later sync:`,
              syncError.message
            );

            // For network errors, record error time after consecutive failures
            if (!syncError.response) {
              if (syncFailCount >= 3) {
                await recordApiErrorTime();
                syncFailCount = 0;
              }
            }

            await addToOfflineQueue(increment);
          }
        }

        lastUpdateTime = now;
      }

      // Always update lastStepCount
      lastStepCount = currentSteps;
    });

    if (typeof options.onSubscriptionSet === "function") {
      options.onSubscriptionSet(subscription);
    }

    return subscription;
  } catch (error) {
    console.error("Error starting pedometer subscription:", error);
    throw error;
  }
};

/**
 * Stops the given pedometer subscription.
 * @param {Object} subscription - The subscription object returned by startPedometerSubscription.
 */
export const stopPedometerSubscription = (subscription) => {
  if (subscription && typeof subscription.remove === "function") {
    subscription.remove();
    console.log("Pedometer subscription stopped");
  } else {
    console.warn("No valid subscription to stop");
  }
};
