// app/utils/stepUtils.js
import AsyncStorage from "@react-native-async-storage/async-storage";
import dayjs from "dayjs";
import NetInfo from "@react-native-community/netinfo";
import { Platform } from "react-native";
import * as stepsEndpoints from "@/app/api/endpoints/steps";
import useAuthStore from "@/app/store/authStore";

/**
 * 1. Network Helpers
 */
export async function isOnline() {
  const state = await NetInfo.fetch();
  return state.isConnected && state.isInternetReachable;
}

/**
 * 2. Local Storage Helpers
 */
function getUserId() {
  const user = useAuthStore.getState().user;
  return user?.id || "";
}
function getDailyStepKey() {
  const userId = getUserId();
  return userId ? `DAILY_STEP_${userId}` : "DAILY_STEP";
}
function getDailyDateKey() {
  const userId = getUserId();
  return userId ? `DAILY_DATE_${userId}` : "DAILY_DATE";
}
function getStepHistoryKey() {
  const userId = getUserId();
  return userId ? `STEP_HISTORY_${userId}` : "STEP_HISTORY";
}
function getLastHistoryFetchKey() {
  const userId = getUserId();
  return userId ? `LAST_HISTORY_FETCH_${userId}` : "LAST_HISTORY_FETCH";
}
function getOfflineQueueKey() {
  const userId = getUserId();
  return userId ? `STEPS_OFFLINE_QUEUE_${userId}` : "STEPS_OFFLINE_QUEUE";
}
function getLastApiErrorTimeKey() {
  return "LAST_API_ERROR_TIME";
}

/**
 * 3. Step History Functions
 */

// Add new function to convert history to array
export function convertHistoryToArray(historyObj, dailyGoal) {
  const historyArray = Object.keys(historyObj).map((key) => {
    const item = historyObj[key];
    return {
      date: key,
      steps: typeof item === "object" ? item.steps : item,
      percentage:
        typeof dailyGoal === "number"
          ? Math.min(
              100,
              Math.round(
                ((typeof item === "object" ? item.steps : item) / dailyGoal) *
                  100
              )
            )
          : 0,
      calories:
        typeof item === "object"
          ? item.calories
          : Math.round((typeof item === "number" ? item : 0) * 0.04),
      distance:
        typeof item === "object"
          ? item.distance
          : Number(((typeof item === "number" ? item : 0) * 0.0007).toFixed(2)),
    };
  });

  // Sort by date (newest first)
  return historyArray.sort((a, b) => dayjs(b.date).diff(dayjs(a.date)));
}

// Update existing saveStepHistory function
export async function saveStepHistoryToLocal(historyArray, stepHistoryKey) {
  try {
    const historyObj = {};

    // Convert array to object format for storage
    historyArray.forEach((item) => {
      historyObj[item.date] = {
        steps: item.steps,
        calories: item.calories || 0,
        distance: item.distance || 0,
        goalAchieved: item.percentage >= 100,
        date: item.date,
      };
    });

    await AsyncStorage.setItem(stepHistoryKey, JSON.stringify(historyObj));
    console.log("Saved step history to local storage");
  } catch (error) {
    console.error("Error saving step history to local:", error);
  }
}

export async function loadLocalStepHistory() {
  try {
    const historyStr = await AsyncStorage.getItem(getStepHistoryKey());
    if (!historyStr) return {};
    return JSON.parse(historyStr);
  } catch (err) {
    console.error("Error loading local step history:", err);
    return {};
  }
}

// Add these constants at the top
const MAX_STEP_INCREMENT = 100; // Maximum reasonable step increment per update
const STEP_COOLDOWN = 2000; // Minimum time (ms) between step updates

// Add new validation functions
export function isValidStepIncrement(currentSteps, newSteps) {
  const increment = newSteps - currentSteps;
  return increment >= 0 && increment <= MAX_STEP_INCREMENT;
}

let lastStepUpdateTime = 0;

export function shouldProcessStepUpdate() {
  const now = Date.now();
  if (now - lastStepUpdateTime < STEP_COOLDOWN) {
    return false;
  }
  lastStepUpdateTime = now;
  return true;
}

// Update saveStepHistory function
export async function saveStepHistory(date, steps, dailyGoal) {
  try {
    const key = getStepHistoryKey();
    const historyStr = await AsyncStorage.getItem(key);
    const history = historyStr ? JSON.parse(historyStr) : {};
    
    // Get previous steps for the day
    const previousSteps = history[date]?.steps || 0;
    
    // Validate step increment
    if (!isValidStepIncrement(previousSteps, steps)) {
      console.warn(`Invalid step increment: ${steps - previousSteps} steps`);
      return false;
    }

    // Continue with saving if validation passes
    const calories = Math.round(steps * 0.04);
    const distance = Number((steps * 0.0007).toFixed(2));
    const goalAchieved = steps >= dailyGoal;

    history[date] = {
      date,
      steps,
      calories,
      distance,
      goalAchieved,
    };

    await AsyncStorage.setItem(key, JSON.stringify(history));
  } catch (error) {
    console.error("Error saving step history:", error);
    return false;
  }
}

export async function saveHistoryToLocal(historyArray) {
  try {
    const historyObj = {};

    // 배열을 객체 형태로 변환하여 저장
    historyArray.forEach((item) => {
      historyObj[item.date] = {
        steps: item.steps,
        calories: item.calories || 0,
        distance: item.distance || 0,
        goalAchieved: item.percentage >= 100,
        date: item.date,
      };
    });

    await AsyncStorage.setItem(stepHistoryKey, JSON.stringify(historyObj));
    console.log("Saved step history to local storage");
  } catch (error) {
    console.error("Error saving step history to local:", error);
  }
}

/**
 * 4. API Sync Helpers
 */
export const API_COOLDOWN = 5 * 60 * 1000; // 5 minutes

export async function isInApiCooldown() {
  try {
    const lastErrorTime = await AsyncStorage.getItem(getLastApiErrorTimeKey());
    if (!lastErrorTime) return false;

    const delta = Date.now() - parseInt(lastErrorTime);
    return delta < API_COOLDOWN;
  } catch (error) {
    console.error("Error checking API cooldown:", error);
    return false;
  }
}
export async function recordApiErrorTime() {
  try {
    await AsyncStorage.setItem(getLastApiErrorTimeKey(), Date.now().toString());
  } catch (error) {
    console.error("Error recording API error time:", error);
  }
}

/**
 * 5. Step Sync Logic
 */
export async function getCurrentStepsFromAPI() {
  return stepsEndpoints.getCurrentSteps();
}

export async function sendStepsIncrementToAPI(increment) {
  return stepsEndpoints.accumulateSteps(increment);
}

export async function createNewRecordInAPI(steps) {
  return stepsEndpoints.sendStepsToDB(steps);
}

/**
 * 6. Offline Queue
 */
export async function addToOfflineQueue(increment) {
  try {
    const queueKey = getOfflineQueueKey();
    const existingQueueStr = await AsyncStorage.getItem(queueKey);
    const existingQueue = existingQueueStr ? JSON.parse(existingQueueStr) : [];

    existingQueue.push({ steps: increment, timestamp: Date.now() });

    await AsyncStorage.setItem(queueKey, JSON.stringify(existingQueue));
    console.log("Added to offline queue:", increment);
  } catch (error) {
    console.error("Error adding to offline queue:", error);
  }
}
export async function getOfflineQueue() {
  try {
    const queueKey = getOfflineQueueKey();
    const queueStr = await AsyncStorage.getItem(queueKey);
    return queueStr ? JSON.parse(queueStr) : [];
  } catch (error) {
    console.error("Error reading offline queue:", error);
    return [];
  }
}
export async function clearOfflineQueue() {
  const queueKey = getOfflineQueueKey();
  await AsyncStorage.setItem(queueKey, JSON.stringify([]));
  console.log("Cleared offline queue.");
}

/**
 * 7. Utility
 */
export async function ensureDailyReset() {
  try {
    const dailyDateKey = getDailyDateKey();
    const storedDate = await AsyncStorage.getItem(dailyDateKey);
    const today = dayjs().format("YYYY-MM-DD");
    if (storedDate !== today) {
      console.log("New day detected, resetting step counter to 0");
      await AsyncStorage.setItem(getDailyDateKey(), today);
      await AsyncStorage.setItem(getDailyStepKey(), "0");
      return true;
    }
    return false;
  } catch (error) {
    console.error("Error ensuring daily reset:", error);
    return false;
  }
}

export async function storeStepsLocally(steps) {
  try {
    const dailyStepKey = getDailyStepKey();
    await AsyncStorage.setItem(dailyStepKey, steps.toString());
  } catch (err) {
    console.error("Error storing steps locally:", err);
  }
}

export async function getStoredSteps() {
  try {
    const dailyStepKey = getDailyStepKey();
    const stepStr = await AsyncStorage.getItem(dailyStepKey);
    return stepStr ? parseInt(stepStr, 10) : 0;
  } catch (err) {
    console.error("Error reading stored steps:", err);
    return 0;
  }
}
