from typing import List
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from app.api import deps
from app.schemas.friendship import FriendshipResponse
from app.services import friendship_service

router = APIRouter()

@router.get("/", response_model=List[FriendshipResponse])
def get_friends(
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    Retrieves the friend list for the currently logged-in user.
    Only accepted friendships are returned. 
    In each friendship record, the friend_id is the ID of the user that is _not_ the current user's ID.
    """
    friends = friendship_service.get_extended_friend_list(db, current_user.id)
    if not friends:
        raise HTTPException(status_code=404, detail="No friends found.")
    return friends

@router.post("/add/{friend_id}", response_model=FriendshipResponse)
def add_friend(
    friend_id: int,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    현재 로그인한 사용자가 friend_id에 해당하는 사용자를 친구로 추가합니다.
    이미 해당 친구 관계가 존재하면 기존 레코드를 반환합니다.
    """
    friendship = friendship_service.create_friendship(
        db, user_id=current_user.id, friend_id=friend_id, status="accepted"
    )
    if not friendship:
        raise HTTPException(status_code=400, detail="Failed to add friend.")
    return friendship

@router.post("/request/{friend_id}", response_model=FriendshipResponse)
def send_friend_request(
    friend_id: int,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    friend_id에 해당하는 사용자에게 친구 요청을 보냅니다.
    요청은 pending 상태로 생성되며, 친구 목록 조회 시에도 출력될 수 있습니다.
    """
    friendship = friendship_service.create_friendship(
        db,
        user_id=current_user.id,
        friend_id=friend_id,
        status="pending"
    )
    if not friendship:
        raise HTTPException(status_code=400, detail="Failed to send friend request.")
    return friendship

@router.put("/accept/{friend_id}", response_model=FriendshipResponse)
def accept_friend_request(
    friend_id: int,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    friend_id에 해당하는 사용자의 친구 요청을 수락합니다.
    상태가 accepted로 업데이트되며, 응답에는 친구의 상세 정보가 포함됩니다.
    """
    friendship_response = friendship_service.update_friendship_status(
        db,
        current_user_id=current_user.id,
        friend_id=friend_id,
        new_status="accepted"
    )
    return friendship_response

@router.put("/reject/{friend_id}", response_model=FriendshipResponse)
def reject_friend_request(
    friend_id: int,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    friend_id에 해당하는 사용자의 친구 요청을 거절합니다.
    (여기서는 거절 상태를 blocked로 처리하도록 구현합니다.)
    """
    friendship = friendship_service.update_friendship_status(
        db,
        current_user_id=current_user.id,
        friend_id=friend_id,
        new_status="blocked"
    )
    return friendship

@router.delete("/{friend_id}", response_model=FriendshipResponse)
def delete_friend(
    friend_id: int,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    friend_id에 해당하는 사용자와의 친구 관계를 'blocked' 상태로 업데이트합니다.
    실제 레코드는 데이터베이스에 남아있어 추후 로그나 분석에 활용할 수 있습니다.
    """
    friendship = friendship_service.delete_friendship(
        db,
        current_user_id=current_user.id,
        friend_id=friend_id
    )
    if not friendship:
        raise HTTPException(status_code=400, detail="Failed to update friendship to blocked")
    return friendship 