from typing import List
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from app.api import deps
from app.schemas.step_record import StepRecordCreate, StepRecordResponse, StepRecordUpdate
from app.services import step_record_service
from app.services.user_service import UserService

router = APIRouter()

@router.post("/", response_model=StepRecordResponse)
def create_step_record_endpoint(
    payload: StepRecordCreate,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    POST /api/v1/steps
    Saves the step data for the logged-in user.
    If a record already exists, it updates it.
    """
    record = step_record_service.create_step_record(db, user_id=current_user.id, step_data=payload)
    return record

@router.post("/steps")
def save_steps(
    user_id: int,
    steps: int,
    db: Session = Depends(deps.get_db)
):
    """
    Example endpoint for saving user steps.
    When a user earns points through their steps, a certain percentage of the earned points
    is awarded to the referrers in the referral chain.
    """
    # Example: 1 step = 0.01 points conversion rate
    conversion_rate = 0.01
    earned_points = int(steps * conversion_rate)
    
    # Award bonus points based on the referral chain
    UserService(db, referred_user_id=user_id, earned_points=earned_points)
    
    # Additional logic for saving step records can be added.
    
    return {"message": "Steps saved and referral bonuses awarded if eligible."}

@router.get("/current", response_model=StepRecordResponse)
def get_current_step_record_endpoint(
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    GET /api/v1/steps/current
    Retrieves the step record for the logged-in user for the current day.
    If no record exists, a 404 error is raised.
    """
    record = step_record_service.get_current_step_record(db, user_id=current_user.id)
    if not record:
        raise HTTPException(
            status_code=404, 
            detail="No step record found for today."
        )
    return record

@router.patch("/accumulate", response_model=StepRecordResponse)
def accumulate_step_record(
    payload: StepRecordUpdate,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    PATCH /api/v1/steps/accumulate
    Accumulates new steps into today's record for the logged-in user.
    If no record exists for today, a new record will be created.
    """
    record = step_record_service.update_step_record_accumulation(
        db,
        user_id=current_user.id,
        steps_increment=payload.steps_increment
    )
    return record

@router.get("/all", response_model=List[StepRecordResponse])
def get_all_step_records_endpoint(
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    GET /api/v1/steps/all
    Retrieves all step records for the logged-in user.
    """
    records = step_record_service.get_all_step_records(db, user_id=current_user.id)
    return records