from fastapi import APIRouter, Depends, HTTPException, status
from sqlalchemy.orm import Session
from typing import List

from app.api import deps
from app.schemas.user_address import AddressCreate, AddressUpdate, AddressResponse
from app.services import address_service

router = APIRouter()

@router.post("/", response_model=AddressResponse, status_code=status.HTTP_201_CREATED)
def create_address(
    address: AddressCreate,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    새 주소를 등록합니다.
    
    이 주소가 기본 주소로 설정되면 다른 모든 주소의 기본 주소 설정이 해제됩니다.
    """
    return address_service.create_address(db, address, current_user.id)

@router.get("/", response_model=List[AddressResponse])
def get_addresses(
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    현재 로그인한 사용자의 모든 주소를 조회합니다.
    """
    return address_service.get_user_addresses(db, current_user.id)

@router.get("/{address_id}", response_model=AddressResponse)
def get_address(
    address_id: int,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    특정 주소를 조회합니다.
    """
    address = address_service.get_address(db, address_id, current_user.id)
    if not address:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail="Address not found"
        )
    return address

@router.put("/{address_id}", response_model=AddressResponse)
def update_address(
    address_id: int,
    address: AddressUpdate,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    주소를 수정합니다.
    이 주소가 기본 주소로 설정되면 다른 모든 주소의 기본 주소 설정이 해제됩니다.
    """
    updated_address = address_service.update_address(db, address_id, address, current_user.id)
    if not updated_address:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail="Address not found"
        )
    return updated_address

@router.delete("/{address_id}", status_code=status.HTTP_204_NO_CONTENT)
def delete_address(
    address_id: int,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    주소를 삭제합니다.    
    삭제되는 주소가 기본 주소였다면, 다른 주소가 기본 주소로 자동 설정됩니다.
    """
    deleted = address_service.delete_address(db, address_id, current_user.id)
    if not deleted:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail="Address not found"
        )
    return None

@router.post("/{address_id}/default", response_model=AddressResponse)
def set_default_address(
    address_id: int,
    db: Session = Depends(deps.get_db),
    current_user = Depends(deps.get_current_user)
):
    """
    특정 주소를 기본 주소로 설정합니다.
    이전에 기본 주소로 설정된 다른 모든 주소의 기본 주소 설정이 해제됩니다.
    """
    address = address_service.set_default_address(db, address_id, current_user.id)
    if not address:
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail="Address not found"
        )
    return address 