from datetime import datetime, timedelta
from typing import Any, Union
from jose import jwt, JWTError
from passlib.context import CryptContext
from app.core.config import settings
from app.core.time import now

pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

# 현재 시간도 동일하게 UTC의 offset-aware datetime으로 변환
current_time = now()  # app.core.time의 now() 사용

def get_password_hash(password: str) -> str:
    return pwd_context.hash(password)

def create_access_token(
    subject: Union[str, Any], expires_delta: timedelta = None
) -> str:
    current_time = now()  # 함수 호출 시마다 최신 시간을 사용합니다.
    if expires_delta:
        expire = current_time + expires_delta
    else:
        expire = current_time + timedelta(minutes=settings.ACCESS_TOKEN_EXPIRE_MINUTES)
    to_encode = {"exp": expire, "sub": str(subject), "scope": "access_token"}
    encoded_jwt = jwt.encode(to_encode, settings.SECRET_KEY, algorithm="HS256")
    return encoded_jwt

def create_refresh_token(subject: Union[str, Any], expires_delta: timedelta = None) -> str:
    current_time = now()
    if expires_delta:
        expire = current_time + expires_delta
    else:
        expire = current_time + timedelta(days=7)
    to_encode = {"exp": expire, "sub": str(subject), "scope": "refresh_token"}
    encoded_jwt = jwt.encode(to_encode, settings.SECRET_KEY, algorithm="HS256")
    return encoded_jwt

def verify_refresh_token(token: str) -> Union[str, None]:
    """
    Verifies that the provided token is a valid refresh token.
    Checks its scope and returns the subject (user id) if valid.
    """
    try:
        payload = jwt.decode(token, settings.SECRET_KEY, algorithms=["HS256"])
        if payload.get("scope") != "refresh_token":
            return None
        return payload.get("sub")
    except JWTError:
        return None

def verify_password(plain_password: str, hashed_password: str) -> bool:
    return pwd_context.verify(plain_password, hashed_password)

def generate_verification_code() -> str:
    """
    Generates a zero-padded 6-digit numeric verification code.
    """
    import random
    return f"{random.randint(0, 999999):06d}"
