from sqlalchemy.orm import Session
import logging

logger = logging.getLogger('adimsayar')

def check_duplicate_after_delete(db: Session, table, condition) -> bool:
    """
    지정된 조건으로 레코드를 삭제한 후, 동일한 조건의 레코드가 다시 나타나는지 확인합니다.
    
    Args:
        db (Session): 데이터베이스 세션
        table: 테이블 모델 클래스
        condition: 삭제할 레코드를 찾기 위한 필터 조건
        
    Returns:
        bool: 삭제 후 동일 조건의 레코드가 존재하면 True, 아니면 False
    """
    try:
        # 레코드 조회
        record = db.query(table).filter(condition).first()
        if not record:
            return False
            
        # 레코드 ID 저장
        record_id = record.id
        
        # 레코드 삭제
        db.delete(record)
        db.commit()
        
        # 동일한 ID로 레코드가 존재하는지 확인
        same_id_check = db.query(table).filter_by(id=record_id).first()
        if same_id_check:
            logger.warning(f"Record with ID {record_id} still exists after deletion!")
            return True
            
        # 다른 ID로 동일한 조건의 레코드가 있는지 확인
        duplicate_check = db.query(table).filter(condition).first()
        if duplicate_check:
            logger.warning(f"Found duplicate record after deletion: Original ID={record_id}, New ID={duplicate_check.id}")
            return True
            
        return False
        
    except Exception as e:
        db.rollback()
        logger.error(f"Error checking duplicates after delete: {str(e)}")
        return False 