from typing import List, Optional
from sqlalchemy.orm import Session
from app.models.device_token import DeviceToken
import logging

logger = logging.getLogger('adimsayar')

class DeviceTokenService:
    @staticmethod
    def get_user_device_tokens(db: Session, user_id: int) -> List[DeviceToken]:
        """사용자의 모든 활성화된 디바이스 토큰을 조회합니다."""
        tokens = db.query(DeviceToken).filter(
            DeviceToken.user_id == user_id,
            DeviceToken.is_active == True
        ).all()
        return tokens

    @staticmethod
    def add_device_token(db: Session, user_id: int, device_token: str, device_type: str) -> DeviceToken:
        """사용자의 디바이스 토큰을 등록하거나 기존 토큰을 활성화합니다."""
        try:
            # 토큰이 Expo 토큰인지 확인
            if device_token.startswith("ExponentPushToken["):
                # Expo 토큰은 플랫폼에 관계없이 'expo'로 표시할 수 있음
                device_type = "expo"
            
            # 이미 존재하는 토큰인지 확인
            existing_token = db.query(DeviceToken).filter(
                DeviceToken.user_id == user_id,
                DeviceToken.device_token == device_token
            ).first()
            
            if existing_token:
                # 기존 토큰이 비활성화 상태라면 활성화
                if not existing_token.is_active:
                    existing_token.is_active = True
                    db.commit()
                    db.refresh(existing_token)
                return existing_token
            
            # 새 토큰 생성
            new_token = DeviceToken(
                user_id=user_id,
                device_token=device_token,
                device_type=device_type,
                is_active=True
            )
            db.add(new_token)
            db.commit()
            db.refresh(new_token)
            logger.info(f"새 디바이스 토큰 등록: 사용자 ID: {user_id}, 디바이스 타입: {device_type}")
            return new_token
        
        except Exception as e:
            logger.error(f"디바이스 토큰 등록 중 오류 발생: {str(e)}")
            raise

    @staticmethod
    def deactivate_device_token(db: Session, user_id: int, device_token: str) -> Optional[DeviceToken]:
        """사용자의 디바이스 토큰을 비활성화합니다."""
        token = db.query(DeviceToken).filter(
            DeviceToken.user_id == user_id,
            DeviceToken.device_token == device_token
        ).first()
        
        if not token:
            logger.warning(f"비활성화할 토큰을 찾을 수 없음: 사용자 ID: {user_id}")
            return None
        
        token.is_active = False
        db.commit()
        db.refresh(token)
        logger.info(f"디바이스 토큰 비활성화: 사용자 ID: {user_id}")
        return token

    @staticmethod
    def get_all_active_tokens(db: Session) -> List[DeviceToken]:
        """모든 활성화된 디바이스 토큰을 조회합니다."""
        return db.query(DeviceToken).filter(DeviceToken.is_active == True).all() 